//=============================================================================
//              |
// Headerfile   |  robo_tx_fw.h
//              |
// Description  |  Header file with definition of the software interface
//              |  to the ROBO TX Controller firmware.
//              |  Can be used for building C-programs which can run
//              |  under control of the ROBO TX Controller firmware in
//              |  download (local) mode or for building PC-programs which
//              |  can communicate with the ROBO TX Controller firmware in
//              |  online mode.
//              |
//-----------------------------------------------------------------------------
// Disclaimer - Exclusion of Liability
//
// This software is distributed in the hope that it will be useful,but WITHOUT 
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
// FITNESS FOR A PARTICULAR PURPOSE. It can be used an modified by anyone
// free of any license obligations or authoring rights.
//=============================================================================

#ifndef __ROBO_TX_FW_H__
#define __ROBO_TX_FW_H__
#define FW_1_24    //my own adaptations to comply with FW version 1.24

#define N_CNT               4           //  number of counters
#define N_PWM_CHAN          8           //  number of PWM channels
#define N_MOTOR             4           //  number of motors
#define N_UNI               8           //  number of universal inputs

// 5kOhm Range
#define R_MIN               10          //  [Ohm]
#define R_MAX               4999        //  [Ohm]
#define R_OVR               5000        //  [Ohm] overload

// 10V Range
#define U_MIN               0           //  [mV]
#define U_MAX               9999        //  [mV]
#define U_OVR               10000       //  [mV] overload

// Ultrasonic Sensor Range
#define ULTRASONIC_MIN      2           //  [cm]
#define ULTRASONIC_MAX      1023        //  [cm]
#define ULTRASONIC_OVR      1024        //  [cm] overload
#define NO_ULTRASONIC       4096        //  not present

// Length of strings
#define DEV_NAME_LEN        16          // "ROBO TX-xxxxxxxx"
#define BLUETOOTH_ADDR_LEN  17          // "xx:xx:xx:xx:xx:xx"
#define DISPL_MSG_LEN_MAX   64          // max length of a pop-up display message


// Identifiers of the Transfer Area parts
enum ta_id_e
{
    TA_LOCAL = 0,           // Local part of Transfer Area. Corresponds to the device on which  
                            // program is currently running in download (local) mode or to the
                            // remotely controlled device (seen from controlled device not from
                            // controlling device) in online mode.
    TA_EXT_1,               // Extension 1 part of Transfer Area
    TA_EXT_2,               // Extension 2 part of Transfer Area
    TA_EXT_3,               // Extension 3 part of Transfer Area
    TA_EXT_4,               // Extension 4 part of Transfer Area
    TA_EXT_5,               // Extension 5 part of Transfer Area
    TA_EXT_6,               // Extension 6 part of Transfer Area
    TA_EXT_7,               // Extension 7 part of Transfer Area
    TA_EXT_8,               // Extension 8 part of Transfer Area
    TA_N_PARTS              // Number of Transfer Area parts
};

#define N_EXT_DEV           (TA_N_PARTS - 1)    // Number of extension devices = 8


// Device functioning modes
enum dev_mode_e
{
    DEV_MODE_LOCAL = 0,
    DEV_MODE_ONLINE,
    DEV_MODE_INVALID
};


// State of connection to an extension device
enum ext_dev_connect_state_e
{
    EXT_DEV_OFFLINE = 0,
    EXT_DEV_ONLINE,
    EXT_DEV_INVALID
};


// Modes of universal inputs
enum input_mode_e
{
    MODE_U = 0,
    MODE_R = 1,
    MODE_ULTRASONIC = 3,
    MODE_INVALID
};


// Program states
enum pgm_state_e
{
    PGM_STATE_INVALID = 0,
    PGM_STATE_RUN,
    PGM_STATE_STOP
};


// Timer units for GetSystemTime hook function
enum TimerUnit
{
    TIMER_UNIT_INVALID        = 0,
    TIMER_UNIT_SECONDS        = 2,
    TIMER_UNIT_MILLISECONDS   = 3,
    TIMER_UNIT_MICROSECONDS   = 4
};


//=============================================================================
//  Structures for Transfer Area (TA)
//=============================================================================

typedef char    BD_ADDR[6]; //leftmost byte in lowest address

// Program information, 8 bytes
typedef struct
{
    char          * name;   // name of a program with a full path, for example, "/ramdisk/Program_1"
    UINT8           state;  // enum pgm_state_e
    char            reserved[3];
} PGM_INFO;


// Display message, 68 bytes. Used to show pop-up message box on the boards display
typedef struct
{
    unsigned char   id; // should be increased by 1 each time a new pop-up message is to be shown
    char            text[DISPL_MSG_LEN_MAX + 1];
    char            reserved[2];
} DISPLAY_MSG;


// Display frame, 8 bytes. Used to refresh boards display with a bitmap image frame
typedef struct
{
    unsigned char * frame;  // contents of a frame as a 128x64 pixels bitmap
    UINT16          id;     // should be increased by 1 each time a new display frame is to be shown
    BOOL16          is_pgm_master_of_display;   // ++ if program wants to have control over display,
                                                // i.e. image frame is displayed over firmware menus;
                                                // -- if program wants to return control over display
                                                // to the firmware menus
} DISPLAY_FRAME;


// Version structure definition, 4 bytes
typedef union
{
    unsigned long abcd;
    struct
    {
        unsigned char   a;
        unsigned char   b;
        unsigned char   c;
        unsigned char   d;
    } part;
} FT_VER;


// Versions of hardware and firmware components, 16 bytes
typedef struct//verified
{
    FT_VER          hardware;   // version of hardware (hardware.part.a = 'A' or 'B' or 'C')
    FT_VER          firmware;   // version of firmware ("V %d.%02d", firmware.part.c, firmware.part.d) firmware.part.b = DLL version
    FT_VER          ta;         // version of transfer area ("V %d.%02d", ta.part.c, ta.part.d)
    char            reserved[4];
} FT_VERSION;


// Info structure, 64 bytes
typedef struct //verified
{
    char            device_name[DEV_NAME_LEN + 1];
    char            bt_addr[BLUETOOTH_ADDR_LEN + 1];
    char            reserved;
    unsigned long   ta_start_addr;
    unsigned long   pgm_area_start_addr;
    unsigned long   pgm_area_size;
    FT_VERSION      version;
} TA_INFO;


// State structure, 36 bytes
//changed in 1.24 to 24 bytes
typedef struct
{
    // Used by local program
//    BOOL8           pgm_initialized;
//    char            reserved_1[7];

    // Public state info
//    BOOL8           dev_mode;   // enum dev_mode_e
//    UCHAR8          id;         // Should be increased by 1 each time something (except id fields)
                                // is changed in this state structure
//    UCHAR8          info_id;    // Should be increased by 1 each time something is changed in info structure
//    UCHAR8          config_id;  // Should be increased by 1 each time something is changed in config structure
    BOOL8           ext_dev_connect_state[N_EXT_DEV];    // enum ext_dev_connect_state_e //verified
    char            reserved_2[8];
    PGM_INFO        local_pgm;  // Corresponds to the program currently being in the program memory
} TA_STATE;


// Universal inputs configuration, 4 bytes
//changed in 1.24 to 1 byte
typedef struct //verified
{
    UINT8           mode;       // enum input_mode_e, for digital inputs msb=1, for analog msb=0
//    BOOL8           digital;    // FALSE = analog input, TRUE = digital input
//    unsigned char   reserved[2];
} UNI_CONFIG;


// Counter inputs configuration, 4 bytes
//changed in 1.24 to 1 byte
typedef struct
{
    UINT8           mode;       // enum input_mode_e
//    unsigned char   reserved[3];
} CNT_CONFIG;


// Config structure, 88 bytes
//changed in 1.24 to 48 bytes
typedef struct
{
    //UINT8           pgm_state_req;  // enum pgm_state_e
    //char            reserved_1[3];
    BOOL8           motor[N_MOTOR];//verified
    UNI_CONFIG      uni[N_UNI];//verified
    CNT_CONFIG      cnt[N_CNT];//verfied 1=normal, 0=inverted
    char            reserved_2[32];
} TA_CONFIG;


// Input structure, 40 bytes (wrong! 44 bytes)
//changed in 1.24 to 48 bytes
typedef struct
{
    INT16           uni[N_UNI];//verified
    BOOL8           cnt_in[N_CNT];//was INT16
    INT16           counter[N_CNT];//verified
    // Number of milliseconds during which the left display button is being kept pressed
    INT16           display_button_left;//verified
    // Number of milliseconds during which the right display button is being kept pressed
    INT16           display_button_right;//verified
    char            unknown[8];//new in 1.24
    // Set to 1 by motor control if target position is reached
    INT16           motor_pos_reached[N_MOTOR];//verified
} TA_INPUT;


// Output structure, 36 bytes
//changed in 1.24 to 44 bytes, don't know what changed, 4 bytes added before duty. 4 after
typedef struct
{
    // Counter reset requests
    INT16           cnt_reset[N_CNT];//was BOOL8
    // If not 0, synchronize this channel with the given channel (1:channel 0, ...)
    UINT8           master[N_MOTOR];
    // Selected motor PWM values
    INT16           duty[N_PWM_CHAN];//verified
    // Selected distane (counter value) at which motor shall stop
    UINT16          distance[N_MOTOR];//verified
    UINT16            reserved[4];//was char
} TA_OUTPUT;


// Display structure, 76 bytes
typedef struct
{
    DISPLAY_MSG     display_msg;
    DISPLAY_FRAME   display_frame;
} TA_DISPLAY;

//new in 1.24 message structure, 24 bytes
typedef struct
{
    UINT16            index;
    UINT16            status;
    UINT16            length;
    UINT8            hwid;
    UINT8            subid;
    UINT16            cmd;
    UINT16            val;
    UINT16            unknown2[6];//possibly another message or room for larger messages
} TA_MESSAGE;

//new in 1.24 message structure, 32 bytes, note: this struct is not preceded by a transferarea-ID
typedef struct
{
    UINT16            result; //??? 1 for the first (empty) result, 2 for each real result, 5 for the last (empty) result.
    BD_ADDR            bd_addr;
    UINT16            unknown;
    char            device_name[DEV_NAME_LEN + 1];
    char            pad[1];
    UINT32            unknown2;
} TA_INQUIRY;

// Change structure (only for ftMscLib), 8 bytes
typedef struct
{
    char            reserved_1[2];
    UINT8           ChangeStatus;
    UINT8           ChangeUni;
    UINT8           ChangeCntIn;
    UINT8           ChangeCounter;
    char            reserved_2[2];
} TA_CHANGE;


// 16-bit timers, 12 bytes
typedef struct
{
    UINT16          Timer1ms;
    UINT16          Timer10ms;
    UINT16          Timer100ms;
    UINT16          Timer1s;
    UINT16          Timer10s;
    UINT16          Timer1min;
} TA_TIMER;


// Hook table with pointers to the firmware functions,
// that can be called by local program, 4 bytes
typedef struct
{
    INT32   (*IsRunAllowed) (void);
    UINT32  (*GetSystemTime)(enum TimerUnit unit);
} TA_HOOK_TABLE;



// ============================================================================
//  Transfer Area (TA) of ROBO TX Controller (one part of TA_N_PARTS)
// ============================================================================
#define RESERVED_1_SIZE  4
#define RESERVED_2_SIZE  28
#define RESERVED_3_SIZE \
    (512 - ( \
    sizeof(TA_INFO)         + \
    sizeof(TA_STATE)        + \
    sizeof(TA_CONFIG)       + \
    sizeof(TA_INPUT)        + \
    sizeof(TA_OUTPUT)       + \
    sizeof(TA_DISPLAY)      + \
    RESERVED_1_SIZE         + \
    sizeof(TA_CHANGE)       + \
    sizeof(TA_TIMER)        + \
    RESERVED_2_SIZE         + \
    sizeof(TA_HOOK_TABLE)     \
    ))


typedef struct
{
    TA_INFO             info;           // info structure
    TA_STATE            state;          // state structure
    TA_CONFIG           config;         // config structure   
    TA_INPUT            input;          // input structure
    TA_OUTPUT           output;         // output structure
    TA_DISPLAY          display;        // display structure

    char                reserved_1[RESERVED_1_SIZE];

    TA_CHANGE           change;         // change structure
    TA_TIMER            timer;          // 16-bit timers

    char                reserved_2[RESERVED_2_SIZE];

    TA_HOOK_TABLE       hook_table;     // hook table with functions pointers

    char                reserved_3[RESERVED_3_SIZE];
} TA;


#endif // __ROBO_TX_FW_H__
