/*
 * DISCLAIMER: I hereby place this code in public domain.
 * This library should be considered not fit for production, and I present no warranties
 * that it is fit for any intent or purpose, use it at your own risk.
 * 
 * Nelson Oliveira
*/

#include "SparkfunSerialLCD.h"
#include "mbed.h"
#include "stdarg.h"
   
SparkfunSerialLCD::SparkfunSerialLCD(PinName tx, int baud, bool clrError): _lcd(tx,NC)
{
    if (clrError)
    {
        SendCmd(Target::Module, Cmd::ResetUnit);
    }
    wait(1);        //Gives the serial module time to wakeup
    setBaud(baud);  //Changes the baud
    cls();          //Clears the screen
    Backlight(29);  //Default backlight
}

void SparkfunSerialLCD::SendCmd(int target, int cmd)
{
    _lcd.putc(target);  //Sends target selection byte
    _lcd.putc(cmd);     //Sends command byte
    wait_ms(100);       //General wait to module/lcd to complete task
}

void SparkfunSerialLCD::print(const char *msg)
{
    _lcd.printf(msg);   //Simple passing to Serial printf with no arguments
}

void SparkfunSerialLCD::printf(const char *msg, ...)
{
    va_list args;               //Standard variable args processing to printf
    va_start(args, msg);    
    _lcd.printf(msg, va_arg(args, char*));
    va_end(args);
}

void SparkfunSerialLCD::setCursor(int column, int row)
{
    //Calculates position and sends command with appropriate values
    SendCmd(Target::LCD, Cmd::CursorPos|((row & 0x1) << 6)|(column % 0x3F));
}

void SparkfunSerialLCD::cls()
{
    SendCmd(Target::LCD, Cmd::Clear);       //According to lcd special bytes list (see SparkfunSerialLCD.h for details)
}  

void SparkfunSerialLCD::Backlight(int bright)
{
    SendCmd(Target::Module, bright+128);    //Adjusts bright to conform to serial module needs and sends instruction
}

void SparkfunSerialLCD::setSplash()
{
    SendCmd(Target::Module, Cmd::SetSplash);//Sets the current text as the splash screen
}

void SparkfunSerialLCD::setBaud(int baud)
{
    //Changes baud, defaults to 9600 in case of invalid value
    switch (baud)
    {
        case 2400:
            SendCmd(Target::Module, Cmd::b2400);
            _lcd.baud(baud);
            break;
        case 4800:
            SendCmd(Target::Module, Cmd::b4800);
            _lcd.baud(baud);
            break;
        case 14400:
            SendCmd(Target::Module, Cmd::b14400);
            _lcd.baud(baud);
            break;
        case 19200:
            SendCmd(Target::Module, Cmd::b19200);
            _lcd.baud(baud);
            break;
        case 38400:
            SendCmd(Target::Module, Cmd::b38400);
            _lcd.baud(baud);
            break;
        default:
            SendCmd(Target::Module, Cmd::b9600);
            _lcd.baud(9600);
            break;
    }
}