/* MAX9723 Driver Library
 * Copyright (c) 2014 Neil Thiessen
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "MAX9723.h"

MAX9723::MAX9723(PinName sda, PinName scl, Address addr, int hz) : m_I2C(sda, scl), m_ADDR((int)addr)
{
    //Initialize the member variables
    m_AmpValue = 255;

    //Set the I2C bus frequency
    m_I2C.frequency(hz);
}

bool MAX9723::open()
{
    //Probe for the MAX9723 using a Zero Length Transfer
    if (!m_I2C.write(m_ADDR, NULL, 0)) {
        //Make sure the amp value variable is set to the default register value
        m_AmpValue = 255;

        //Write the default register value
        m_I2C.write(m_ADDR, &m_AmpValue, 1);

        //Return success
        return true;
    } else {
        //Return failure
        return false;
    }
}

bool MAX9723::enabled()
{
    //Return the status of the SHUTDOWN bit
    return (m_AmpValue & (1 << 7));
}

void MAX9723::enabled(bool enabled)
{
    //Set or clear the SHUTDOWN bit
    if (enabled)
        m_AmpValue |= (1 << 7);
    else
        m_AmpValue &= ~(1 << 7);

    //Write the value back out
    m_I2C.write(m_ADDR, &m_AmpValue, 1);
}

bool MAX9723::bassMax()
{
    //Return the status of the BassMax ENABLE bit
    return (m_AmpValue & (1 << 6));
}

void MAX9723::bassMax(bool enabled)
{
    //Set or clear the BassMax ENABLE bit
    if (enabled)
        m_AmpValue |= (1 << 6);
    else
        m_AmpValue &= ~(1 << 6);

    //Write the value back out
    m_I2C.write(m_ADDR, &m_AmpValue, 1);
}

bool MAX9723::maxGain()
{
    //Return the status of the MAXIMUM GAIN bit
    return (m_AmpValue & (1 << 5));
}

void MAX9723::maxGain(bool max)
{
    //Set or clear the MAXIMUM GAIN bit
    if (max)
        m_AmpValue |= (1 << 5);
    else
        m_AmpValue &= ~(1 << 5);

    //Write the value back out
    m_I2C.write(m_ADDR, &m_AmpValue, 1);
}

float MAX9723::volume()
{
    //Return the bottom 5 bits
    return (m_AmpValue & 0x1F) / 31.0;
}

void MAX9723::volume(float volume)
{
    //Mask off the old volume
    m_AmpValue &= 0xE0;

    //Set the new volume
    m_AmpValue |= (char)(volume * 31.0);

    //Write the value back out
    m_I2C.write(m_ADDR, &m_AmpValue, 1);
}

#ifdef MBED_OPERATORS
MAX9723::operator float()
{
    //Return the current volume
    return volume();
}

MAX9723& MAX9723::operator=(float value)
{
    //Set the volume
    volume(value);
    return *this;
}
#endif
