/* MAX9723 Driver Library
 * Copyright (c) 2014 Neil Thiessen
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef MAX9723_H
#define MAX9723_H

#include "mbed.h"

/** MAX9723 class.
 *  Used for controlling a MAX9723 headphone amplifier connected via I2C (TODO: Redo Example!).
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "MAX9723.h"
 *
 * //Create a MAX9723 object at the default address (ADDRESS_0)
 * MAX9723 amp(p28, p27);
 *
 * int main()
 * {
 *     //Try to open the MAX9723
 *     if (amp.open()) {
 *         printf("Device detected!\n");
 *
 *         //Configure the MAX9723 for no BassMax, and low gain
 *         amp.enabled(true);
 *         amp.bassMax(false);
 *         amp.maxGain(false);
 *
 *         //Set the volume for 50%
 *         amp = 0.50;
 *     } else {
 *         error("Device not detected!\n");
 *     }
 * }
 * @endcode
 */
class MAX9723
{
public:
    /** Represents the different I2C address possibilities for the MAX9723
     */
    enum Address {
        ADDRESS_0 = (0x4C << 1),    /**< MAX9723A or MAX9723C */
        ADDRESS_1 = (0x4D << 1)     /**< MAX9723B or MAX9723D */
    };

    /** Create an MAX9723 object connected to the specified I2C pins with the specified I2C slave address
     *
     * @param sda The I2C data pin.
     * @param scl The I2C clock pin.
     * @param addr The I2C slave address (defaults to ADDRESS_0).
     * @param hz The I2C bus frequency (defaults to 400kHz).
     */
    MAX9723(PinName sda, PinName scl, Address addr = ADDRESS_0, int hz = 400000);

    /** Probe for the MAX9723 and set it to default values if present
     *
     * @returns
     *   'true' if the device exists on the bus,
     *   'false' if the device doesn't exist on the bus.
     */
    bool open();

    /** Determine whether or not the MAX9723 is enabled
     *
     * @returns
     *   'true' if the MAX9723 is enabled,
     *   'false' if the MAX9723 is shutdown.
     */
    bool enabled();

    /** Set whether or not the MAX9723 is enabled
     *
     * @param enabled Whether or not the MAX9723 is enabled.
     */
    void enabled(bool enabled);

    /** Determine whether or not BassMax is enabled on the MAX9723
     *
     * @returns
     *   'true' if BassMax is enabled,
     *   'false' if BassMax is disabled.
     */
    bool bassMax();

    /** Enable or disable BassMax on the MAX9723
     *
     * @param enabled Whether or not BassMax is enabled.
     */
    void bassMax(bool enabled);

    /** Determine whether or not the MAX9723 gain is set to maximum
     *
     * @returns
     *   'true' if the gain is set to 0dB on MAX9723A and MAX9723B, or +6dB on MAX9723C and MAX9723D,
     *   'false' if the gain is set to -5dB on MAX9723A and MAX9723B, or +1dB on MAX9723C and MAX9723D.
     */
    bool maxGain();

    /** Set whether or not the MAX9723 gain is set to maximum
     *
     * @param max Whether or not the gain is set to maximum.
     */
    void maxGain(bool max);

    /** Get the current volume of the MAX9723 as a percentage
     *
     * @returns The current volume as a percentage (0.0 to 1.0).
     */
    float volume();

    /** Set the volume of the MAX9723 from a percentage
     *
     * @param volume The new volume as a percentage (0.0 to 1.0).
     */
    void volume(float volume);

#ifdef MBED_OPERATORS
    /** A shorthand for volume()
     *
     * @returns The current volume as a percentage (0.0 to 1.0).
     */
    operator float();

    /** A shorthand for volume()
     *
     * @param value The new volume as a percentage (0.0 to 1.0).
     */
    MAX9723& operator=(float value);
#endif

private:
    //I2C member variables
    I2C m_I2C;
    const int m_ADDR;

    //Amplifier settings member variables
    char m_AmpValue;
};

#endif
