/**
 *  \brief CS294-84 demo \author Ben Zhang, Antonio Iannopollo
 *
 * This sampel code illustrates how to connect the mbed KL25Z platform to internet
 * thorugh the CC3000 wifi breakout board (http://www.adafruit.com/product/1469).
 * Connections between the KL25Z and the CC3000 are made according to the
 * guide at https://learn.adafruit.com/adafruit-cc3000-wifi -- KL25Z and arduino
 * UNO are pin to pin compatible --
 *
 * This application uses the following libraries:
 * - cc3000_hostdriver_mbedsocket
 *   (http://developer.mbed.org/users/Kojto/code/cc3000_hostdriver_mbedsocket/)
 * - HTTPClient (http://developer.mbed.org/users/donatien/code/HTTPClient/)
 */

#include <string>

#include "cc3000.h"
#include "mbed.h"
#include "HTTPClient.h"

const string POLLING_URL = "http://universe.eecs.berkeley.edu:8080/value";
const string UPDATE_URL = "http://universe.eecs.berkeley.edu:8080/update?value=";

// KL25Z wifi connection
// we need to define connection pins for:
// - IRQ      => (pin D3)
// - Enable   => (pin D5)
// - SPI CS   => (pin D10)
// - SPI MOSI => (pin D11)
// - SPI MISO => (pin D12)
// - SPI CLK  => (pin D13)
// plus wifi network SSID, password, security level and smart-configuration flag.
mbed_cc3000::cc3000 wifi(D3, D5, D10, SPI(D11, D12, D13),
                         "SSID", "PASSWORD", WPA2, false);

// create an http instance
HTTPClient http;

// str is used to hold the response data
char str[512];
char url[80];

// setup the serial connection, and LEDs
Serial pc(USBTX, USBRX);
DigitalOut led_green(LED_GREEN);
PwmOut led_red(LED_RED);

Ticker reportSensorTicker;
bool shouldReadSensor;
int sensorData;

void readSensor() {
    // avoid doing much work here (in the interrupt).
    // put the actual sensor reading in the main function
    shouldReadSensor = true;
}

int main() {
    pc.printf("CC3000 Nodejs Demo Starts\r\n");
    // by default, we make the LED red
    led_red = 0;
    led_green = 1;

    // reset the readsensor bool to false so we don't read the data.
    shouldReadSensor = false;
    sensorData = 0;
    // Every 5 second, we sample the sensor and report the data.
    reportSensorTicker.attach(&readSensor, 5.0); 
    
    wifi.init();

    while(1) {
        // continuosly check connection status
        if(wifi.is_connected() == false) {
            // try to connect
            if (wifi.connect() == -1) {
                pc.printf("Failed to connect."
                          "Please verify connection details and try again.\r\n");
                continue;
            } else {

                pc.printf("IP address: %s \r\n", wifi.getIPAddress());
                // once connected, turn green LED on and red LED off
                led_red = 1;
                led_green = 0;
            }
        }

        // update the value when shouldReadSensor is true          
        if (shouldReadSensor == true) {
            shouldReadSensor = false;
            sensorData += 10;
            char url[128];
            snprintf(url, sizeof(url), "%s%d", UPDATE_URL.c_str(), sensorData);
            int ret = http.get(url, str, 128);
            if (!ret) {
                pc.printf("sending sensorData %d, response: %s\r\n", sensorData, str);
            } else {
                pc.printf("[PUSH] Error - ret = %d - HTTP %d\r\n", ret, http.getHTTPResponseCode());
            }
        }
        
        // everytime we do HTTP polling
        int ret = http.get(POLLING_URL.c_str(), str, 128);
        if (!ret) {
            int value = atoi(str);
            led_red = 1.0 * value / 255;
        } else {
            pc.printf("[POLL] Error - ret = %d - HTTP %d\r\n", ret, http.getHTTPResponseCode());
        }
        wait(0.1);
    }
}
