#include "mbed.h"
#include "hcsr04.h"
#include "ESP8266.h"
#include "math.h"
#include "FXOS8700Q.h"
#include "FXAS21000.h"
#include "RegisterWriter/RegisterWriter/rohm_hal2.h"
#include "rohm-bh1790glc-driver/bh1790glc_registers.h"
#include "rohm-bh1790glc-driver/bh1790glc.h"

#define CloudIP "184.106.153.149"           //Raw IP Address of ThingSpeak Cloud Server
#define SSID "SSID"
#define Password "Password"

HCSR04 usensor1(D10,D11);                   //ECHO Pin=D11, TRIG Pin=D10
Serial pc(USBTX,USBRX);                     //Serial Communication with PC
ESP8266 wifi(PTC17, PTC16, 115200);         //Tx Pin:PTC17; Rx Pin:PTC17; Baud rate:115200

I2C i2c(PTE25, PTE24);
FXOS8700QAccelerometer acc(i2c, FXOS8700CQ_SLAVE_ADDR1);    // Configured for the FRDM-K64F with onboard sensors
FXOS8700QMagnetometer mag(i2c, FXOS8700CQ_SLAVE_ADDR1);
FXAS21000 gyro(D14,D15);

I2C i2chm(I2C_SDA, I2C_SCL);
RegisterWriter i2c_rw(i2chm);
BH1790GLC bh1790glc(i2c_rw);

DigitalOut buzzer(D7);

int on = 1, off = 0; 

void wifi_send(void);               //Connect and Push Data Channel to Cloud Server
void US_Sensor(void);               //Take ultrasonic measurement and send to cloud
void Acc_Mag_Gyro(void);            //Take measurements from acceleromter, magnetometer, and gyroscope and send to cloud
void Heart_Monitor(void);              

int num = 0;
char snd[255],rcv[1000];                    //snd: send command to ESP8266, rcv: receive response from ESP8266
    
//Ultrasonic sensor variable
int distance1 = 100;
    
//acc, mag, and gryo variables
float gyro_data[3];
motion_data_units_t acc_data, mag_data;
motion_data_counts_t acc_raw, mag_raw;
float faX, faY, faZ, fmX, fmY, fmZ, tmp_float;
int16_t raX, raY, raZ, rmX, rmY, rmZ, tmp_int;

//heart monitor variables
bool interval_elapsed;
Ticker ticker;
uint16_t data[2];

void timer_isr()
{
    interval_elapsed = true;
}

int main()
{
    buzzer = off;
    pc.baud(115200);                        //Baud Rate of 115200 for Tera Term
    
    acc.enable();
    mag.enable();

    pc.baud(115200);
    i2c.frequency(400000);
    
    int error;

    wait(0.1);

    do
    { //until init OK.
        error = bh1790glc.set_default_on();
        wait_ms(500);
        wait_ms(200);
    }
    while (error);
    
    bh1790glc.set_default_on();
    
    interval_elapsed = false;
    ticker.attach(&timer_isr, 0.03125);                 //32Hz
    
    pc.printf("Initial Setup\r\n");
    wifi.SetMode(1);                        //Set ESP mode to 1
    wifi.RcvReply(rcv, 1000);               //Receive a response from ESP
    pc.printf("%s\r", rcv);

    pc.printf("Connecting to WiFi\r\n");    //AP Setup Initialization
    wifi.Join(SSID, Password);              //Put your Wifi SSID followed by Password
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\n", rcv);
    wait(2);
    
    wifi.GetIP(rcv);                        //Obtains an IP address from the AP
    
    while (1) 
    {
        wifi_send();
        
        wait(1.0f);
    }
}

void wifi_send(void)
{
    while(num<1000000000000)
    {
        num=num+1;
        pc.printf("\nSyncing to Thingspeak #: %d\n\r", num);
        
        US_Sensor();
        
        Acc_Mag_Gyro();     
        
        Heart_Monitor();
    }
}

void US_Sensor(void)
{
    //Ultrasound Sensor (HC-SR04) #1 Initialization
    usensor1.start();
    wait_ms(100);
        
    //Calculating Distance Percentage Remaining for Sensor # 1
    distance1 = usensor1.get_dist_cm();
        
    //Sending Data to the Cloud Server via ESP8266 WiFi Module
    strcpy(snd,"AT+CIPMUX=0\n\r");        //AT+CIPMUX: Enabling Single Channel Mode
    wifi.SendCMD(snd);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    wait(0.1);
        
    //Establish TCP connection w/ Cloud Server
    sprintf(snd,"AT+CIPSTART=4,\"TCP\",\"%s\",80\n",CloudIP);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    wait(0.1);
        
    //Set length of the data that will be sent
    strcpy(snd,"AT+CIPSEND=100\n\r");
    wifi.SendCMD(snd);
    pc.printf("%s\r", rcv);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\r", rcv);
    wait(0.1);
        
    //Pushing the data acquired from HC-SR04 Ultrasonic Sensor to Cloud Server via API
    sprintf(snd,"\rhttps://api.thingspeak.com/update?api_key=MR7SIH7MSC5KIIH0&field1=%d\r", distance1);
    printf("Distance from Object: %dcm\n\r", distance1);
    
    
    
    if(distance1 < 6)
    {
        buzzer = on;
        wait(.3);
        buzzer = off;
        wait(0.3);
        buzzer = on;
        wait(0.3);
        buzzer = off;
        wait(0.3);
        buzzer = on;
        wait(0.3);
        buzzer = off;
    }
    
    wifi.SendCMD(snd);
    pc.printf("%s\r",snd);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\r", rcv);    
}

void Acc_Mag_Gyro(void)
{
    //counts based results
    acc.getAxis(acc_raw);
    mag.getAxis(mag_raw);
    acc.getX(raX);
    acc.getY(raY);
    acc.getZ(raZ);
    mag.getX(rmX);
    mag.getY(rmY);
    mag.getZ(rmZ);
            
    //unit based results
    acc.getAxis(acc_data);
    mag.getAxis(mag_data);
    acc.getX(faX);
    acc.getY(faY);
    acc.getZ(faZ);
    mag.getX(fmX);
    mag.getY(fmY);
    mag.getZ(fmZ);
    pc.printf("Acellerometer: X=%1.4fY=%1.4fZ=%1.4f\n\r", acc.getX(faX),acc.getY(faY),acc.getZ(faZ));
    pc.printf("Magnetometer: X=%4.1fY=%4.1fZ=%4.1f\n\r", mag.getX(fmX),mag.getY(fmY), mag.getZ(fmZ));
    gyro.ReadXYZ(gyro_data);
    pc.printf("Gyro: X=%6.2f Y=%6.2f Z=%6.2f\n\r", gyro_data[0],gyro_data[1], gyro_data[2]);
    wait(0.5f);
            
    //Sending Data to the Cloud Server via ESP8266 WiFi Module
    strcpy(snd,"AT+CIPMUX=0\n\r");        //AT+CIPMUX: Enabling Single Channel Mode
    wifi.SendCMD(snd);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    wait(0.1);
        
    //Establish TCP connection w/ Cloud Server
    sprintf(snd,"AT+CIPSTART=4,\"TCP\",\"%s\",80\n",CloudIP);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    wait(0.1);
        
    //Set length of the data that will be sent
    strcpy(snd,"AT+CIPSEND=100\n\r");
    wifi.SendCMD(snd);
    pc.printf("%s\r", rcv);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\r", rcv);
    wait(0.1);
        
    //Pushing the data acquired from Acc to Cloud Server via API
    //Sends only X data for simplicity of testing purposes
    sprintf(snd,"\rhttps://api.thingspeak.com/update?api_key=MR7SIH7MSC5KIIH0&field2=%f\r", acc.getX(faX));
    printf("Current X Acceleration: %f\n\r", acc.getX(faX));
    wifi.SendCMD(snd);
    pc.printf("%s\r",snd);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\r", rcv);   
    
    //Pushing the data acquired from Mag to Cloud Server via API
    //Sends only X data for simplicity of testing purposes
    sprintf(snd,"\rhttps://api.thingspeak.com/update?api_key=MR7SIH7MSC5KIIH0&field3=%f\r", mag.getX(fmX));
    printf("Current X Position: %f\n\r", mag.getX(fmX));
    wifi.SendCMD(snd);
    pc.printf("%s\r",snd);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\r", rcv); 
    
    //Pushing the data acquired from Gyro to Cloud Server via API
    //Sends only X data for simplicity of testing purposes
    sprintf(snd,"\rhttps://api.thingspeak.com/update?api_key=MR7SIH7MSC5KIIH0&field4=%f\r", gyro_data[0]);
    printf("Current X Orientation: %f\n\r", gyro_data[0]);
    wifi.SendCMD(snd);
    pc.printf("%s\r",snd);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\r", rcv);  
}

void Heart_Monitor(void)
{
    int i = 0;
    int sum = 0;
    int count = 0;
    float average = 0;
    int error;
    
    wait_ms(500);
    
    while(i < 10)
        { 
            if (interval_elapsed)       //Wait until ISR
            {                         
                error = bh1790glc.getresults(data);
                
                if (!error) 
                {                          
                    //get data, print to serial, update flags
                    pc.printf("%d, \t%d\n\r", data[1], data[0]);                
                    interval_elapsed = false;
                    i++;
                    sum += data[1];
                    count++;
                }   
            }
        }
        
        average = sum/10;
        
    if(average > 3500)
    {
        buzzer = on;
        wait(.1);
        buzzer = off;
        wait(0.1);
        buzzer = on;
        wait(0.1);
        buzzer = off;
        wait(0.1);
        buzzer = on;
        wait(.1);
        buzzer = off;
        wait(0.1);
        buzzer = on;
        wait(0.1);
        buzzer = off;
        wait(0.1);
        buzzer = on;
        wait(.1);
        buzzer = off;
        wait(0.1);
        buzzer = on;
        wait(0.1);
        buzzer = off;
    }
        
    //Sending Data to the Cloud Server via ESP8266 WiFi Module
    strcpy(snd,"AT+CIPMUX=0\n\r");        //AT+CIPMUX: Enabling Single Channel Mode
    wifi.SendCMD(snd);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    wait(0.1);
        
    //Establish TCP connection w/ Cloud Server
    sprintf(snd,"AT+CIPSTART=4,\"TCP\",\"%s\",80\n",CloudIP);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    wait(0.1);
        
    //Set length of the data that will be sent
    strcpy(snd,"AT+CIPSEND=100\n\r");
    wifi.SendCMD(snd);
    pc.printf("%s\r", rcv);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\r", rcv);
    wait(0.1);
        
    //Pushing the data acquired from heart monitor to Cloud Server via API
    //Sends only X data for simplicity of testing purposes
    sprintf(snd,"\rhttps://api.thingspeak.com/update?api_key=MR7SIH7MSC5KIIH0&field5=%f\r", average);
    printf("Current average heart rate: %f\n\r", average);
    wifi.SendCMD(snd);
    pc.printf("%s\r",snd);
    wait(0.1);
    wifi.RcvReply(rcv, 1000);
    pc.printf("%s\r", rcv);
}
