#pragma once
#include "mbed.h"
#include "math.h"
#include "MBed_Adafruit_GPS.h"
#include "SDFileSystem.h"

#include "QAM.h"
#include "dataPoint.h"

#define SAMPLE_LENGTH       1000 
#define SAMPLE_RATE         10000
#define SIN_LENGTH          500
#define OUTAVG_LENGTH       100
#define CARRIER_FREQ        220
#define TIME_CONST          0.0001
#define PI                  3.14159265

SDFileSystem sd(PTE3, PTE1, PTE2, PTE4, "sd"); // MOSI, MISO, SCK, CS
FILE *fp;
char buffer[1024];
 
Serial * gps_Serial;

Ticker tick1;
Ticker tick2;
AnalogIn AnLong(A0);
AnalogIn AnShort(A1);
AnalogIn AnRefLong(A2);
AnalogIn AnRefShort(A3);
AnalogOut dac0(DAC0_OUT);
DigitalOut led_red(LED_RED);
Serial pc(USBTX, USBRX);

int sinRes = (int)1/(CARRIER_FREQ*TIME_CONST); 
int freq = 1;

float sinWave[SIN_LENGTH] = {};
int sinIndex = 0;

float sLQ[SAMPLE_LENGTH] = {};
float sLI[SAMPLE_LENGTH] = {};
float sSQ[SAMPLE_LENGTH] = {};
float sSI[SAMPLE_LENGTH] = {};
float sRefLQ[SAMPLE_LENGTH] = {};
float sRefLI[SAMPLE_LENGTH] = {};
float sRefSQ[SAMPLE_LENGTH] = {};
float sRefSI[SAMPLE_LENGTH] = {};

float Iarray[SAMPLE_LENGTH] = {};
float Qarray[SAMPLE_LENGTH] = {};
int sampleIndex = 0;
float I = 0;
float Q = 0;
float lon = 0;
float lonRef = 0;
float shor = 0;
float shorRef = 0;

float decimateLong[4] = {};
float decimateShort[4] = {};
float decimateRef[4] = {};

float totalAVG = 0;

bool isSampling = true;
int avgIndex = 0;

char * timeDatestamp = "";


void print_array(float *bar,int length, FILE *file){
    int i =0;
    for(i = 0; i < length; i++){
        fprintf(file, "%f, ", bar[i]);
    }
}

void print_array_serial(float *bar,int length){
    int i =0;
    for(i = 0; i < length; i++){
        pc.printf("%f, ", bar[i]);
    }
}

float avg_array(float *array, int length){
    int i = 0;
    float total = 0;
    for(i = 0; i < length; i++){
        total += array[i];
    }   
    return total/length;
}

void buildIQ(){
    for(int i = 0; i < SAMPLE_LENGTH; i++){
        Iarray[i] = cos(2*PI*CARRIER_FREQ*i*TIME_CONST);
        Qarray[i] = -sin(2*PI*CARRIER_FREQ*i*TIME_CONST);
    }
}

void tick_out(){
    if(isSampling){
        dac0 = sinWave[sinIndex];
        sinIndex++;
        if((sinIndex+1) > sinRes){
            sinIndex = 0;
        }
        
        lon = AnLong.read();
        lonRef = AnRefLong.read();
        shor = AnShort.read();
        shorRef = AnRefShort.read();
        
        I = Iarray[sampleIndex];
        Q = Qarray[sampleIndex];
        sLI[sampleIndex] = lon*I;
        sLQ[sampleIndex] = lon*Q;
        sSI[sampleIndex] = shor*I;
        sSQ[sampleIndex] = shor*Q;
        sRefLI[sampleIndex] = lonRef*I;
        sRefLQ[sampleIndex] = lonRef*Q;
        sRefSI[sampleIndex] = shorRef*I;
        sRefSQ[sampleIndex] = shorRef*Q;
        
        sampleIndex++;
        if(sampleIndex+1 > SAMPLE_LENGTH){
            sampleIndex--;
        }
        
        
    }

}

void create_sinWave(){
    int i = 0;
    for(i = 0; i < sinRes; i++){
        sinWave[i] = 0.25 * sin(2.0*PI*i/sinRes) + 0.75;
    }
}

void set_Values(int inFreq){
    freq = inFreq;
    create_sinWave();
}

void print_values(){
    fp = fopen("/sd/data.txt", "r");
    if (fp != NULL) {
        fclose(fp);
        remove("/sd/hello.txt");
        pc.printf("Remove an existing file with the same name \n");
    }

    printf("\nWriting data to the sd card \n");
    fp = fopen("/sd/hello.txt", "w");
    if (fp == NULL) {
        pc.printf("Unable to write the file \n");
    }
    fclose(fp);
}

void save_point(char * c){
    fp = fopen("/sd/data.txt", "r");
    if (fp != NULL) {
        fclose(fp);
        remove("/sd/data.txt");
        pc.printf("Remove an existing file with the same name \n");
    }

    printf("\nWriting data to the sd card \n");
    fp = fopen("/sd/data.txt", "w");
    if (fp == NULL) {
        pc.printf("Unable to write the file \n");
    } else {
        fprintf(fp, c);
        fprintf(fp, "\n");
        fclose(fp);
    }
}

int main(){
    pc.baud(115200);
    
    gps_Serial = new Serial(D1,D0); //serial object for use w/ GPS
    Adafruit_GPS myGPS(gps_Serial); //object of Adafruit's GPS class
    char c; //when read via Adafruit_GPS::read(), the class returns single character stored here
    Timer refresh_Timer; //sets up a timer for use in loop; how often do we print GPS info?
    Timer sampling_Toggle;
    Timer t; 
    const int refresh_Time = 1000; //refresh time in ms
    const int sample_Time = 250;
    const int filter_Time = 2000;
    
    myGPS.begin(9600);  //sets baud rate for GPS communication; note this may be changed via Adafruit_GPS::sendCommand(char *)
                        //a list of GPS commands is available at http://www.adafruit.com/datasheets/PMTK_A08.pdf
    
    myGPS.sendCommand(PMTK_SET_NMEA_OUTPUT_RMCGGA); //these commands are defined in MBed_Adafruit_GPS.h; a link is provided there for command creation
    myGPS.sendCommand(PMTK_SET_NMEA_UPDATE_1HZ);
    myGPS.sendCommand(PGCMD_ANTENNA);
    
    buildIQ();
    
    tick1.attach(&tick_out, 0.0001); // below 0.00005 the board can no longer output and read
    //tick2.attach(&print_values, 20);
    
    set_Values(220);
    pc.printf("sinres = %i\n", sinRes);
    
    float filteredLongTemp = 0;
    float filteredShortTemp = 0;
    float filteredRefLongTemp = 0;
    float filteredRefShortTemp = 0;
    
    refresh_Timer.start();  
    sampling_Toggle.start();
    
    while(1){        
        if(isSampling){
            if (sampling_Toggle.read_ms() >= sample_Time) {
                isSampling = false;
                sampling_Toggle.reset();
                
                filteredLongTemp = qam_in(sLI, sLQ, &pc);
                filteredRefLongTemp = qam_in(sRefLI, sRefLQ, &pc);
                filteredShortTemp = qam_in(sSI, sSQ, &pc);
                filteredRefShortTemp = qam_in(sRefSI, sRefSQ, &pc);
                pc.printf("Long = %f,  Short = %f ",filteredLongTemp, filteredShortTemp);
                
                fp = fopen("/sd/data.txt", "w");
                if (fp == NULL) {
                    pc.printf("Unable to write the file \n");
                } else {
                    fprintf(fp,"Long = %f,  Short = %f ",filteredLongTemp/filteredRefShortTemp, filteredShortTemp/filteredRefShortTemp);
                    fclose(fp);
                }
            }
        }else{
            if (sampling_Toggle.read_ms() >= filter_Time) {
                isSampling = true;
                sampling_Toggle.reset();
                
                sampleIndex = 0;
            }                
        } 
        
        if(!isSampling){  
            c = myGPS.read();   
            if ( myGPS.newNMEAreceived() ) {
                if ( !myGPS.parse(myGPS.lastNMEA()) ) {
                    continue;   
                }    
            }
        
            //check if enough time has passed to warrant printing GPS info to screen
            //note if refresh_Time is too low or pc.baud is too low, GPS data may be lost during printing
            if (refresh_Timer.read_ms() >= refresh_Time) {
                refresh_Timer.reset();
                pc.printf("Time: %d:%d:%d.%u \n", myGPS.hour, myGPS.minute, myGPS.seconds, myGPS.milliseconds);
                fp = fopen("/sd/data.txt", "w");
                if (fp == NULL) {
                    pc.printf("Unable to write the file \n");
                } else {
                    fprintf(fp,"Time: %d:%d:%d.%u \n", myGPS.hour, myGPS.minute, myGPS.seconds, myGPS.milliseconds);
                    fclose(fp);
                }   
                if (myGPS.fix) {
                    pc.printf("Location: %5.2f%c, %5.2f%c\n", myGPS.latitude, myGPS.lat, myGPS.longitude, myGPS.lon);
                    fp = fopen("/sd/data.txt", "w");
                    if (fp == NULL) {
                        pc.printf("Unable to write the file \n");
                    } else {
                        fprintf(fp,"Location: %5.2f%c, %5.2f%c\n", myGPS.latitude, myGPS.lat, myGPS.longitude, myGPS.lon);
                        fclose(fp);
                    }
                }
            }
        }                  
    }
}