#include "private.h"
#include <string.h>

/*
 * common constants
 */
#define BAUD_RATE 115200
//#define BAUD_RATE 9600
#define CR  13
#define LF  10
#define READ_BUFF_LEN       4096
#define IPADDRESS_LENGTH    16
#define PORT_NUM    "1080"

/*
 * mbed pin settings
 */
Serial wifi(p13, p14);
DigitalOut PRST(p15);

class WifiTerminalMode {
private:
    unsigned char buff[READ_BUFF_LEN];
    unsigned char myIPAddress[IPADDRESS_LENGTH];

private:
    void putc(int c)
    {
        while (!wifi.writeable())
            ;   // empty loop body
        wifi.putc(c);
    }

    int getc()
    {
        return wifi.getc();
    }

    void delayedPutc(unsigned char c)
    {
        wait(0.00032);
        putc(c);
    }

    void write(const unsigned char *data)
    {
        const unsigned char *p;
        for ( p = data; *p != '\0'; p++ ) {
            delayedPutc(*p);
        }
    }

    int readToPrompt()
    {
        int len;
        unsigned char *p = buff;
        for ( len = 0; len < READ_BUFF_LEN - 1; len++, p++ ) {
            *p = getc();
            if ( *p == '#' ) {
                break;
            }
        }
        *p = '\0';

        return len;
    }

    void waitPrompt()
    {
        while (getc() != '#')
            ;   // empty loop body
    }

public:
    unsigned char *readBlock()
    {
        int len;
        unsigned char *p = buff;
        for ( len = 0; len < READ_BUFF_LEN - 1; len++ ) {
            *p++ = getc();
            if ( !wifi.readable() ) {
                break;
            }
        }
        *p = '\0';

        return buff;
    }

    unsigned char *readLine()
    {
        int len;
        unsigned char *p = buff;
        
        for ( len = 0; len < READ_BUFF_LEN - 1; len++, p++ ) {
            *p = getc();
            if ( *p == CR || *p == LF ) {
                p++;
                break;
            }
        }
        *p = '\0';

        return buff;
    }
private:
    int command(const unsigned char *data)
    {
        int len = 0;
        unsigned char *p2 = buff;
        const unsigned char *p;
    
        for ( p = data; *p != '\0'; p++ ) {
            while (wifi.readable()) {
                *p2++ = getc();
                len++;
            }
            delayedPutc(*p);
        }
        for (; len < READ_BUFF_LEN - 1; len++, p2++ ) {
            *p2 = getc();
            if ( *p2 == '#') {
                break;
            }
        }
        *p2 = '\0';

        return len;
    }

public:
    void reset()
    {
        wifi.baud(BAUD_RATE);
        wifi.format(8, Serial::None, 1);
    
        PRST = 0;       // activate reset line
        wait(1.0);    // perhaps needs 1 sec
        PRST = 1;       // deactivate reset line
    }

    void serialInit()
    {
        int i;
        int c = 0;

        while (true) {
            if (wifi.writeable())
                putc('A');
            if (wifi.readable()) {
                if ( c == '*' ) {
                    c = getc();
                    if ( c == CR ) {
                        break;
                    }
                } else {
                    c = getc();
                }
            }
            wait(0.00032);  // this wait is important
        }
        // flush buffer
        while (wifi.readable())
            getc();
        // change to config mode
        for ( i = 0; i < 8; i++ ) {
            delayedPutc(' ');
        }
        c = 0;
        while (true) {
            if (wifi.readable()) {
                if ( c == '*' ) {
                    c = getc();
                    if ( c == CR ) {
                        break;
                    }
                } else {
                    c = getc();
                }
            }
        }
    }

    void portSetup()
    {
        command("wlan_type set infra\r");
        command("wlan_ssid set " SSID "\r");
        command("wlan_wps set stop\r");
        command("wlan_crdl set off\r");
#ifdef  WEP40_KEY
        command("wlan_sec set wep40\r");
        command("wlan_wep set " WEP40_KEY "\r");
#endif
#ifdef  WEP104_KEY
        command("wlan_sec set wep104\r");
        command("wlan_wep set " WEP104_KEY "\r");
#endif
#ifdef  TKIP_KEY
        command("wlan_sec set wpa-tkip\r");
        command("wlan_psk set " TKIP_KEY "\r");
#endif
#ifdef  AES_KEY
        command("wlan_sec set wpa2-aes\r");
        command("wlan_psk set " AES_KEY "\r");
#endif
#ifdef  MIX_KEY
        command("wlan_sec set wpa-mix\r");
        command("wlan_psk set " MIX_KEY "\r");
#endif
        command("ip_dhcp set on\r");
        command("ip_term_prot set tcps\r");
        command("ip_term_hp set " PORT_NUM "\r");
        command("save permit\r");
    }
    
    unsigned char *getAddr()
    {
        int len;
        int i = 0;
        const char *p;
    
        waitPrompt();
        while (true) {
            len = command("wlan_con get\r");
            if (len > 0 && strstr((const char *)buff, (const char *)"= Connect") != 0 ) {
                break;
            }
            setLeds(4 | i);
            i ^= 1;
            wait(1.0);
        }
        do {
            setLeds(6 | i);
            i ^= 1;
            wait(1.0);
            len = command("ip_current get\r");
            if ( len <= 0 ) {
                p = (const char *)buff;
                buff[0] = '\0';
                continue;
            }
            p = strstr((const char *)buff, (const char *)"IP");
            if ( p == NULL )
                continue;
            p = strstr(p, (const char *)" = ");
            if ( p == NULL )
                continue;
        } while ( strncmp(p, " = 0.0.0.0", 10) == 0 );
        p += 3;
        for ( i = 0; i < IPADDRESS_LENGTH - 1; i++, p++ ) {
            if ( *p == '\r' )
                break;
            myIPAddress[i] = *p;
        }
        myIPAddress[i] = '\0';
        command("run permit\r");
    
        return myIPAddress;
    }
};