/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: Target board general functions implementation

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Miguel Luis and Gregory Cristian
*/
#include "mbed.h"
#include "board.h"

MoteVersion_t BoardGetVersion( void );

DigitalOut RedLed( PB_1 );     // Active Low
DigitalOut YellowLed( PB_10 ); // Active Low
DigitalOut GreenLed( PC_3 );   // Active Low
DigitalOut UsrLed( PA_5 );     // Active High


GPS Gps( PB_6, PB_7, PB_11 ); // Gps(tx, rx, en);

DigitalIn I2cInterrupt( PB_4 );
I2C I2c(I2C_SDA, I2C_SCL);

MPL3115A2 Mpl3115a2( I2c, I2cInterrupt );
MMA8451Q Mma8451q(I2c, I2cInterrupt);

DigitalOut Pc7( PC_7 );
DigitalIn Pc1( PC_1 );

// Used for Push button application demo
DigitalIn PC0( PC_0, PullUp ); 

AnalogIn *Battery;

#define AIN_VREF            3.3     // STM32 internal refernce
#define AIN_VBAT_DIV        2       // Resistor divider

SX1272MB2xAS Radio( NULL );

void BoardInit( void )
{
    // Initalize LEDs
    RedLed = 1;     // Active Low
    GreenLed = 1;   // Active Low
    YellowLed = 1; // Active Low
    UsrLed = 0;     // Active High

    TimerTimeCounterInit( );

    switch( BoardGetVersion( ) )
    {
        case MOTE_VERSION_2:
            Battery = new AnalogIn( PA_0 );
            Gps.en_invert = true;
            break;
        case MOTE_VERSION_3:
            Battery = new AnalogIn( PA_1 );
            Gps.en_invert = false;
            break;
        default:
            break;
    }
    Gps.init( );
    Gps.enable( 1 );
    
    Mpl3115a2.init( );
    Mma8451q.orient_detect( );
}


uint8_t BoardGetBatteryLevel( void ) 
{
    // Per LoRaWAN spec; 0 = Charging; 1...254 = level, 255 = N/A
    return ( Battery->read_u16( ) >> 8 ) + ( Battery->read_u16( ) >> 9 );
}

float BoardGetBatteryVoltage( void ) 
{
    return ( Battery->read( ) * AIN_VREF * AIN_VBAT_DIV );
}

uint32_t BoardGetRandomSeed( void )
{
    return ( ( *( uint32_t* )ID1 ) ^ ( *( uint32_t* )ID2 ) ^ ( *( uint32_t* )ID3 ) );
}

void BoardGetDevEUI( uint8_t *id )
{
    uint32_t *pDevEuiHWord = ( uint32_t* )&id[4];

    if( *pDevEuiHWord == 0 )
    {        
        *pDevEuiHWord = BoardGetRandomSeed( );
    }
    
}

void BoardGetUniqueId( uint8_t *id )
{
    id[7] = ( ( *( uint32_t* )ID1 )+ ( *( uint32_t* )ID3 ) ) >> 24;
    id[6] = ( ( *( uint32_t* )ID1 )+ ( *( uint32_t* )ID3 ) ) >> 16;
    id[5] = ( ( *( uint32_t* )ID1 )+ ( *( uint32_t* )ID3 ) ) >> 8;
    id[4] = ( ( *( uint32_t* )ID1 )+ ( *( uint32_t* )ID3 ) );
    id[3] = ( ( *( uint32_t* )ID2 ) ) >> 24;
    id[2] = ( ( *( uint32_t* )ID2 ) ) >> 16;
    id[1] = ( ( *( uint32_t* )ID2 ) ) >> 8;
    id[0] = ( ( *( uint32_t* )ID2 ) );
}

MoteVersion_t BoardGetVersion( void )
{
    Pc7 = 1;
    char first = Pc1;
    Pc7 = 0;
    
    if( first && !Pc1 )
    {
        return MOTE_VERSION_2;
    }
    else
    {
        return MOTE_VERSION_3;
    }
}
