#pragma once

#include "mbed.h"
#include <vector>
#include <utility>

class InterruptMultiplexer {
    const PinName INTERRUPT_STATUS_PIN = NC;

public:

    InterruptMultiplexer(PinName irq_pin, PinName status_pin = D2)
        : irq(irq_pin), status(INTERRUPT_STATUS_PIN) {
        if (status.is_connected())
            status = 1;
    }

    void clear() {
        isr_array.clear();
    }

    InterruptIn& getIRQ() {
        return irq;
    }

    void enableCallback(int index) {
        if (index < isr_array.size()) {
            isr_array[index].second = true;
        }
    }

    void disableCallback(int index) {
        if (index < isr_array.size()) {
            isr_array[index].second = false;
        }
    }

    void trigger() {
        if (status.is_connected()) {
            if (status)
                status = 0;
            else
                status = 1;
        }
        for (int i = 0; i < isr_array.size(); ++i) {
            bool enable = isr_array[i].second;
            if (enable) {
                FunctionPointer& fptr = isr_array[i].first;
                fptr.call();
            }
        }
    }

    int addCallback(void (*isr)(void), bool enable = true) {
        FunctionPointer fptr;
        fptr.attach(isr);
        isr_array.push_back(std::make_pair(fptr, enable));
        return isr_array.size() - 1;
    }

    template <typename T>
    int addCallback(T* tptr, void (T::*isr)(void), bool enable = true) {
        FunctionPointer fptr;
        fptr.attach(tptr, isr);
        isr_array.push_back(std::make_pair(fptr, enable));
        return isr_array.size() - 1;
    }

private:
    InterruptIn irq;
    std::vector<std::pair<FunctionPointer, bool> > isr_array;
    DigitalOut status;
};
