/* mbed Microcontroller Library - MemFileSystem
 * Copyright (c) 2008, sford
 */


#ifndef MBED_MEMFILESYSTEM_H
#define MBED_MEMFILESYSTEM_H

#include "FATFileSystem.h"
#include "SDFileSystem.h"

#define NUM_OF_SECTORS 2000
#define SECTOR_SIZE    512

namespace mbed
{

    class MemFileSystem : public FATFileSystem
    {
    public:
    
        // 2000 sectors, each 512 bytes (malloced as required)
        char *sectors[NUM_OF_SECTORS];
    
        MemFileSystem(const char* name) : FATFileSystem(name) {
            memset(sectors, 0, sizeof(sectors));
        }
    
        virtual ~MemFileSystem() {
            for(int i = 0; i < NUM_OF_SECTORS; i++) {
                if(sectors[i]) {
                    free(sectors[i]);
                }
            }
        }

        // read a sector in to the buffer, return 0 if ok
        virtual int disk_read(uint8_t *buffer, uint64_t sector, uint8_t count) {
            if(sectors[sector] == 0) {
                // nothing allocated means sector is empty
                memset(buffer, 0, SECTOR_SIZE);
            } else {
                memcpy(buffer, sectors[sector], SECTOR_SIZE);
            }
            return 0;
        }
    
        // write a sector from the buffer, return 0 if ok
        virtual int disk_write(const uint8_t *buffer, uint64_t sector, uint8_t count) {
            // if buffer is zero deallocate sector
            char zero[SECTOR_SIZE];
            memset(zero, 0, SECTOR_SIZE);
            if(memcmp(zero, buffer, SECTOR_SIZE)==0) {
                if(sectors[sector] != 0) {
                    free(sectors[sector]);
                    sectors[sector] = 0;
                }
                return 0;
            }
            // else allocate a sector if needed, and write
            if(sectors[sector] == 0) {
                char *sec = (char*)malloc(SECTOR_SIZE);
                if(sec==0) {
                    return 1; // out of memory
                }
                sectors[sector] = sec;
            }
            memcpy(sectors[sector], buffer, SECTOR_SIZE);
            return 0;
        }
    
        // return the number of sectors
        virtual uint64_t disk_sectors() {
            return sizeof(sectors)/sizeof(sectors[0]);
        }

        void dump(FILE *fp) {
            for (int i = 0; i < NUM_OF_SECTORS; i++) {
                fwrite(&sectors[i], sizeof(int), 1, fp);
                if (sectors[i] != 0) {
                    fwrite(sectors[i], sizeof(char), SECTOR_SIZE, fp);
                }
            }
        }

        void load(FILE *fp) {
            int sec_info = 0;
            for (int i = 0; i < NUM_OF_SECTORS; i++) {
                fread(&sec_info, sizeof(int), 1, fp);
                if (sec_info != 0) {
                    char *sec = (char *)malloc(SECTOR_SIZE);
                    fread(sec, sizeof(char), SECTOR_SIZE, fp);
                    sectors[i] = sec;
                }
            }
        }
    };
}

#endif