#include "SDFileSystem.h"
#include "rtos.h"
#include "mbed.h"

typedef struct uFMT_STRUCT {
  short comp_code;
  short num_channels;
  unsigned voice_rate;
  unsigned avg_Bps;
  short block_align;
  short sig_bps;
} FMT_STRUCT;
 
class wave_player {

public:
/** Create a wave player using a pointer to the given AnalogOut object.
 *
 * @param _dac pointer to an AnalogOut object to which the voices are sent.
 */
    wave_player(AnalogOut *_dac);
    void play(FILE *wavefile);
    void set_verbosity(int v);

/** the player function.
 *
 * @param wavefile  A pointer to an opened wave file
 */

/** Set the printf verbosity of the wave player.  A nonzero verbosity level
 * will put wave_player in a mode where the complete contents of the wave
 * file are echoed to the screen, including header values, and including
 * all of the voice values placed into the DAC FIFO, and the voice values
 * removed from the DAC FIFO by the ISR.  The voice output frequency is
 * fixed at 2 Hz in this mode, so it's all very slow and the DAC output isn't
 * very useful, but it lets you see what's going on and may help for debugging
 * wave files that don't play correctly.
 *
 * @param v the verbosity level
 */
private:
    void dac_out(void);
    int verbosity;
    AnalogOut *wave_DAC;
    Ticker tick;
    unsigned short DAC_fifo[256];
    short DAC_wptr;
    volatile short DAC_rptr;
    short DAC_on;
};

wave_player::wave_player(AnalogOut *_dac)
{
  wave_DAC=_dac;
  wave_DAC->write_u16(32768);        //DAC is 0-3.3V, so idles at ~1.6V
  verbosity=0;
}

//-----------------------------------------------------------------------------
// if verbosity is set then wave player enters a mode where the wave file
// is decoded and displayed to the screen, including voice values put into
// the DAC FIFO, and values read out of the DAC FIFO by the ISR.  The DAC output
// itself is so slow as to be unusable, but this might be handy for debugging
// wave files that don't play
//-----------------------------------------------------------------------------
void wave_player::set_verbosity(int v)
{
  verbosity=v;
}

//-----------------------------------------------------------------------------
// player function.  Takes a pointer to an opened wave file.  The file needs
// to be stored in a filesystem with enough bandwidth to feed the wave data.
// LocalFileSystem isn't, but the SDcard is, at least for 22kHz files.  The
// SDcard filesystem can be hotrodded by increasing the SPI frequency it uses
// internally.
//-----------------------------------------------------------------------------
void wave_player::play(FILE *wavefile)
{
        unsigned chunk_id,chunk_size,channel;
        unsigned data,samp_int,i;
        short unsigned dac_data;
        long long slice_value;
        char *slice_buf;
        short *data_sptr;
        unsigned char *data_bptr;
        int *data_wptr;
        FMT_STRUCT wav_format;
        long slice,num_slices;
  DAC_wptr=0;
  DAC_rptr=0;
  for (i=0;i<256;i+=2) {
    DAC_fifo[i]=0;
    DAC_fifo[i+1]=3000;
  }
  DAC_wptr=4;
  DAC_on=0;

  fread(&chunk_id,4,1,wavefile);
  fread(&chunk_size,4,1,wavefile);
  while (!feof(wavefile)) {
    if (verbosity)
      printf("Read chunk ID 0x%x, size 0x%x\n",chunk_id,chunk_size);
    switch (chunk_id) {
      case 0x46464952:
        fread(&data,4,1,wavefile);
        if (verbosity) {
          printf("RIFF chunk\n");
          printf("  chunk size %d (0x%x)\n",chunk_size,chunk_size);
          printf("  RIFF type 0x%x\n",data);
        }
        break;
      case 0x20746d66:
        fread(&wav_format,sizeof(wav_format),1,wavefile);
        if (verbosity) {
          printf("FORMAT chunk\n");
          printf("  chunk size %d (0x%x)\n",chunk_size,chunk_size);
          printf("  compression code %d\n",wav_format.comp_code);
          printf("  %d channels\n",wav_format.num_channels);
          printf("  %d voices/sec\n",wav_format.voice_rate);
          printf("  %d bytes/sec\n",wav_format.avg_Bps);
          printf("  block align %d\n",wav_format.block_align);
          printf("  %d bits per voice\n",wav_format.sig_bps);
        }
        if (chunk_size > sizeof(wav_format))
          fseek(wavefile,chunk_size-sizeof(wav_format),SEEK_CUR);
        break;
      case 0x61746164:
// allocate a buffer big enough to hold a slice
        slice_buf=(char *)malloc(wav_format.block_align);
        if (!slice_buf) {
          printf("Unable to malloc slice buffer");
          exit(1);
        }
        num_slices=chunk_size/wav_format.block_align;
        samp_int=1000000/(wav_format.voice_rate);
        if (verbosity) {
          printf("DATA chunk\n");
          printf("  chunk size %d (0x%x)\n",chunk_size,chunk_size);
          printf("  %d slices\n",num_slices);
          printf("  Ideal voice interval=%d\n",(unsigned)(1000000.0/wav_format.voice_rate));
          printf("  programmed interrupt tick interval=%d\n",samp_int);
        }

// starting up ticker to write voices out -- no printfs until tick.detach is called
        if (verbosity)
          tick.attach_us(this,&wave_player::dac_out, 500000); 
        else
          tick.attach_us(this,&wave_player::dac_out, samp_int); 
        DAC_on=1; 

// start reading slices, which contain one voice each for however many channels
// are in the wave file.  one channel=mono, two channels=stereo, etc.  Since
// mbed only has a single AnalogOut, all of the channels present are averaged
// to produce a single voice value.  This summing and averaging happens in
// a variable of type signed long long, to make sure that the data doesn't
// overflow regardless of voice size (8 bits, 16 bits, 32 bits).
//
// note that from what I can find that 8 bit wave files use unsigned data,
// while 16 and 32 bit wave files use signed data
//
        for (slice=0;slice<num_slices;slice+=1) {
          fread(slice_buf,wav_format.block_align,1,wavefile);
          if (feof(wavefile)) {
            printf("Oops -- not enough slices in the wave file\n");
            exit(1);
          }
          data_sptr=(short *)slice_buf;     // 16 bit voices
          data_bptr=(unsigned char *)slice_buf;     // 8 bit voices
          data_wptr=(int *)slice_buf;     // 32 bit voices
          slice_value=0;
          for (channel=0;channel<wav_format.num_channels;channel++) {
            switch (wav_format.sig_bps) {
              case 16:
                if (verbosity)
                  printf("16 bit channel %d data=%d ",channel,data_sptr[channel]);
                slice_value+=data_sptr[channel];
                break;
              case 32:
                if (verbosity)
                  printf("32 bit channel %d data=%d ",channel,data_wptr[channel]);
                slice_value+=data_wptr[channel];
                break;
              case 8:
                if (verbosity)
                  printf("8 bit channel %d data=%d ",channel,(int)data_bptr[channel]);
                slice_value+=data_bptr[channel];
                break;
            }
          }
          slice_value/=wav_format.num_channels;
          
// slice_value is now averaged.  Next it needs to be scaled to an unsigned 16 bit value
// with DC offset so it can be written to the DAC.
          switch (wav_format.sig_bps) {
            case 8:     slice_value<<=8;
                        break;
            case 16:    slice_value+=32768;
                        break;
            case 32:    slice_value>>=16;
                        slice_value+=32768;
                        break;
          }
          dac_data=(short unsigned)slice_value;
          if (verbosity)
            printf("voice %d wptr %d slice_value %d dac_data %u\n",slice,DAC_wptr,(int)slice_value,dac_data);
          DAC_fifo[DAC_wptr]=dac_data;
          DAC_wptr=(DAC_wptr+1) & 0xff;
          while (DAC_wptr==DAC_rptr) {
          }
        }
        DAC_on=0;
        tick.detach();
        free(slice_buf);
        break;
      case 0x5453494c:
        if (verbosity)
          printf("INFO chunk, size %d\n",chunk_size);
        fseek(wavefile,chunk_size,SEEK_CUR);
        break;
      default:
        printf("unknown chunk type 0x%x, size %d\n",chunk_id,chunk_size);
        data=fseek(wavefile,chunk_size,SEEK_CUR);
        break;
    }
    fread(&chunk_id,4,1,wavefile);
    fread(&chunk_size,4,1,wavefile);
  }
}

BusOut myleds(LED1,LED2,LED3,LED4);
SDFileSystem sd(p5, p6, p7, p8, "sd");
AnalogOut speaker(p18);
wave_player waver(&speaker);
Serial pc(USBTX, USBRX);

DigitalOut P15(p15);
DigitalOut P16(p16);
DigitalOut P19(p19);
DigitalOut P20(p20);

unsigned short data;


class microphone
{
public :
    microphone(PinName pin);
    unsigned short read_u16();
    operator unsigned short ();
private :
    AnalogIn _pin;
};
microphone::microphone (PinName pin):
    _pin(pin)
{
}
unsigned short microphone::read_u16()
{
    return _pin.read_u16();
}
inline microphone::operator unsigned short ()
{
    return _pin.read_u16();
}
 
 
microphone mymicrophone(p17);

 
int main()
{
    while(1) {
        FILE *wave_file;
        wave_file=fopen("/sd/wavfiles/Yakko's World.wav","r");
        waver.play(wave_file);
        fclose(wave_file);      
    }
}

void wave_player::dac_out()
{
  if (DAC_on) {
#ifdef VERBOSE
  printf("ISR rdptr %d got %u\n",DAC_rptr,DAC_fifo[DAC_rptr]);
#endif
    wave_DAC->write_u16((mymicrophone + DAC_fifo[DAC_rptr])/2);
    DAC_rptr=(DAC_rptr+1) & 0xff;
  }
}