/// @file Watchdog.cpp provides the interface to the Watchdog module
///
/// This provides basic Watchdog service for the mbed. You can configure
/// various timeout intervals that meet your system needs. Additionally,
/// it is possible to identify if the Watchdog was the cause of any 
/// system restart.
/// 
/// Adapted from Simon's Watchdog code from http://mbed.org/forum/mbed/topic/508/
///
/// @note Copyright &copy; 2011 by Smartware Computing, all rights reserved.
///     This software may be used to derive new software, as long as
///     this copyright statement remains in the source file.
/// @author David Smart
///
/// \li v2.10 - 20160914: Changed TARGET_STM by mutech, t.kuroki

/// 2020-01-15 mbed-os5 compatible routones added. by mutech, t.luroki

#include "Watchdog.h"

//------------------------------------------------------------------------------
#if defined( TARGET_LPC1768 )
/// Watchdog gets instantiated at the module level
Watchdog::Watchdog()
{
    _wdreset = (LPC_WDT->WDMOD >> 2) & 1;    // capture the cause of the previous reset
}

// mbed-os5 compatible routines
bool Watchdog::start(uint32_t timeout_ms)
{
    LPC_WDT->WDCLKSEL = 0x1;                // Set CLK src to PCLK
    uint32_t clk = SystemCoreClock / 1000;  // 
    LPC_WDT->WDTC = (timeout_ms * clk) / 16; // WD has a fixed /4 prescaler, PCLK default is /4
    LPC_WDT->WDMOD = 0x3;                   // Enabled and Reset
    kick();
    return true;
}

bool Watchdog::stop(void)
{
    return false;
}

void Watchdog::kick(void)
{
    LPC_WDT->WDFEED = 0xAA;
    LPC_WDT->WDFEED = 0x55;
}

// deprecated routines
/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(float s)
{
    start((uint32_t)(s * 1000.0f));
}

void Watchdog::Configure(int ms)
{
    start(static_cast<uint32_t>(ms));
}

/// "Service", "kick" or "feed" the dog - reset the watchdog timer
/// by writing this required bit pattern
void Watchdog::Service(void)
{
    kick();
}

/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset(void)
{
    return _wdreset;
}

#elif defined(TARGET_LPC4088)
// from Gesotec Gesotec
/// Watchdog gets instantiated at the module level
Watchdog::Watchdog()
{
    _wdreset = (LPC_WDT->MOD >> 2) & 1;    // capture the cause of the previous reset
}
 
// mbed-os5 compatible routines
bool Watchdog::start(uint32_t timeout_ms)
{
    //LPC_WDT->CLKSEL = 0x1;                // Set CLK src to PCLK
    uint32_t clk = 500000 / 4;    // WD has a fixed /4 prescaler, and a 500khz oscillator
    LPC_WDT->TC = (timeout_ms * clk) / 1000;
    LPC_WDT->MOD = 0x3;                   // Enabled and Reset
    kick();
    return true;
}

bool Watchdog::stop(void)
{
    return false;
}

void Watchdog::kick(void)
{
    LPC_WDT->FEED = 0xAA;
    LPC_WDT->FEED = 0x55;
}

// deprecated routines
/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(float s)
{
    start((uint32_t)(s * 1000.0f));
}

void Watchdog::Configure(int ms)
{
    start(static_cast<uint32_t>(ms));
}

/// "Service", "kick" or "feed" the dog - reset the watchdog timer
/// by writing this required bit pattern
void Watchdog::Service(void)
{
    kick();
}
 
/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset(void)
{
    return _wdreset;
}

#elif defined(TARGET_LPC81X) || defined(TARGET_LPC82X)

// from Gesotec Gesotec
/// Watchdog gets instantiated at the module level
Watchdog::Watchdog()
{
    _wdreset = (LPC_WWDT->MOD >> 2) & 1;    // capture the cause of the previous reset
}
 
// mbed-os5 compatible routines
bool Watchdog::start(uint32_t timeout_ms)
{
#if 0
    uint32_t clk = get_wdtclock() / 4;    // WD has a fixed /4 prescaler, and a 500khz oscillator
    LPC_WWDT->TC = (timeout_ms * clk) / 1000;
#else
    LPC_SYSCON->WDTOSCCTRL = WDTOSCCTRL_Val(10, 2);    // wdt_osc_clk = Fclkana/2, Fclkana = 3.5MHz
    LPC_SYSCON->SYSAHBCLKCTRL |= (1 << 17); // Enable Clock WWDT
    LPC_SYSCON->PDRUNCFG &= ~(1 << 6);      // Enable Power WDTOSC_PD
    uint32_t clk = ((3500000/2)/4);         // COUNT = wdt_osc_clk/4
    LPC_WWDT->TC = (timeout_ms * clk) / 1000;
#endif
    LPC_WWDT->MOD = 0x3;                    // Enabled and Reset
    kick();
    return true;
}

bool Watchdog::stop(void)
{
    return false;
}

void Watchdog::kick(void)
{
    LPC_WWDT->FEED = 0xAA;
    LPC_WWDT->FEED = 0x55;
}

// deprecated routines
/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(float s)
{
    start((uint32_t)(s * 1000.0f));
}

#define WDTOSCCTRL_Val(clk, div)    ((((uint32_t)(clk)) << 5) | (((div) >> 1) - 1))

void Watchdog::Configure(int ms)
{
    start(static_cast<uint32_t>(ms));
}

uint32_t Watchdog::get_wdtclock(void)
{
#if 0
    uint32_t wdt_osc = 0;

    /* Determine clock frequency according to clock register values             */
    switch ((LPC_SYSCON->WDTOSCCTRL >> 5) & 0x0F)
    {
        case 0:  wdt_osc =       0; break;
        case 1:  wdt_osc =  600000; break;
        case 2:  wdt_osc = 1050000; break;
        case 3:  wdt_osc = 1400000; break;
        case 4:  wdt_osc = 1750000; break;
        case 5:  wdt_osc = 2100000; break;
        case 6:  wdt_osc = 2400000; break;
        case 7:  wdt_osc = 2700000; break;
        case 8:  wdt_osc = 3000000; break;
        case 9:  wdt_osc = 3250000; break;
        case 10: wdt_osc = 3500000; break;
        case 11: wdt_osc = 3750000; break;
        case 12: wdt_osc = 4000000; break;
        case 13: wdt_osc = 4200000; break;
        case 14: wdt_osc = 4400000; break;
        case 15: wdt_osc = 4600000; break;
    }
#else
    const uint32_t osctab[16] =
    {
              0,    //  0
         600000,    //  1
        1050000,    //  2
        1400000,    //  3
        1750000,    //  4
        2100000,    //  5
        2400000,    //  6
        2700000,    //  7
        3000000,    //  8
        3250000,    //  9
        3500000,    // 10
        3750000,    // 11
        4000000,    // 12
        4200000,    // 13
        4400000,    // 14
        4600000     // 15
    };
#endif
    uint32_t wdt_osc = osctab[(LPC_SYSCON->WDTOSCCTRL >> 5) & 0x0F];

//    wdt_osc /= ((LPC_SYSCON->WDTOSCCTRL & 0x1F) << 1) + 2;
    return wdt_osc;
}

/// "Service", "kick" or "feed" the dog - reset the watchdog timer
/// by writing this required bit pattern
void Watchdog::Service(void)
{
    kick();
}
 
/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset(void)
{
    return _wdreset;
}

#elif defined(TARGET_STM)
Watchdog::Watchdog()
{
    _rcc_csr = RCC->CSR;
    RCC->CSR |= RCC_CSR_RMVF; // clear reset flag
}

// 整数Xを含む最小のべき乗指数
int Watchdog::calcExponent16bit(uint16_t v)
{
//  return (v == 0) ? 0 : MSB16bit(v - 1) + 1;
    if (!v)
        return 0;
    --v;
// 最大有効ビット数（MSB：Most Significant Bit）
    v |= (v >> 1);
    v |= (v >> 2);
    v |= (v >> 4);
    v |= (v >> 8);
//  return count16bit(v) - 1;
// 立っているビットの数を数える
    v = (v & 0x5555) + ((v >> 1) & 0x5555);
    v = (v & 0x3333) + ((v >> 2) & 0x3333);
    v = (v & 0x0f0f) + ((v >> 4) & 0x0f0f);
    return  (v & 0x00ff) + ((v >> 8) & 0x00ff);
}

#if defined(TARGET_STM32F0) || defined(TARGET_STM32F1) || defined(TARGET_STM32F3)
  #define WDT_CLOCK     40000U      // 40 kHz
#else
// TARGET_STM32L0/STM32L1/STM32L4/STM32F4/STM32F7
  #define WDT_CLOCK     32768U      // 32.768 kHz
#endif

// mbed-os5 compatible routines
bool Watchdog::start(uint32_t timeout_ms)
{
    // http://www.st.com/web/en/resource/technical/document/reference_manual/CD00171190.pdf

    // Newer Nucleo boards have 32.768 kHz crystal. Without it, the internal 
    // RC clock would have an average frequency of 40 kHz (variable between 30 and 60 kHz)
//  tick = (ms / (1/WDT_CLOCK))/1000;
    uint32_t tick = ((uint32_t)timeout_ms * WDT_CLOCK + 500U) / 1000U;
    // The RLR register is 12 bits and beyond that a prescaler should be used
    int scale = calcExponent16bit((tick + 4095) >> 12);
    if (scale < 2)
        scale = 2;
    else if (scale > 8)         // STM32 allows a maximum time of around 26.2 seconds for the Watchdog timer
        scale = 8;

    int residual = tick / (1 << scale);   // The value for the RLR register
    if (residual < 1)
        residual = 1;
    else if (residual > 4096)
        residual = 4096;

    IWDG->KR  = 0x5555;         // enable write to PR, RLR
    IWDG->PR  = scale - 2;      // Prescaler has values of multiples of 4 (i.e. 2 ^2), page 486 Reference Manual
    IWDG->RLR = residual - 1;   // Init RLR
    IWDG->KR  = 0xAAAA;         // Reload the watchdog
    IWDG->KR  = 0xCCCC;         // Starts the WD
    return true;
}

bool Watchdog::stop(void)
{
    return false;
}

void Watchdog::kick(void)
{
    IWDG->KR  = 0xAAAA;
}

// deprecated routines
/// Load timeout value in watchdog timer and enable
void Watchdog::Configure(float s)
{
    start((uint32_t)(s * 1000.0f));
}

void Watchdog::Configure(int ms)
{
    start(static_cast<uint32_t>(ms));
}

/// "Service", "kick" or "feed" the dog - reset the watchdog timer
void Watchdog::Service()
{
    kick();
}

/// get the flag to indicate if the watchdog causes the reset
bool Watchdog::WatchdogCausedReset()
{
#if defined(RCC_CSR_IWDGRSTF)
    return (_rcc_csr & (RCC_CSR_IWDGRSTF | RCC_CSR_WWDGRSTF)) != 0;  // read the IWDGRSTF (Independent WD, not the windows WD)
#else
// for old library
    return (_rcc_csr & (RCC_CSR_WDGRSTF | RCC_CSR_WWDGRSTF)) != 0;  // read the IWDGRSTF (Independent WD, not the windows WD)
#endif
}

#endif
//------------------------------------------------------------------------------
