#ifndef STATS_H
#define STATS_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Friendly.h"

/** Friendly Class
@brief Used for drawing the weapons bar, the health bar, and the wave counter in the Nemesis game.
@brief Constantly checks and redraws elements based on user input and progress in the game.
@brief Incorporates N5110.h file by Craig A. Evans.

@brief Revision 1.0

@author Musallam M. M. Bseiso
@date   3rd May 2017
*/


class Stats
{

public:
    
    /// Constructors and destructors:
    Stats();
    ~Stats();
    
    
    ////////////////////////////////
    //////// PUBLIC METHODS
    ////////////////////////////////
    
    
    /** Draw Grid
    *   
    *   Draws the outer rectangular border, health border, wave counter border, and weapon border onto the LCD.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    */
    void draw_grid(N5110 &lcd);
    
    
    /** Draw Wave Counter
    *   
    *   Draws the wave counter onto the LCD.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param wave_counter - counter that stores which wave the game is in
    */
    void draw_wave_counter(N5110 &lcd, int wave_counter);
    
    
    /** Draw Health
    *   
    *   Draws the outside border of the health bar heart onto the LCD.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    */
    void draw_health(N5110 &lcd);
    
    
    /** Draw First Rocket
    *   
    *   Draws the first rocket from the left in the weapons bar onto the LCD. It is drawn full black or dotted
    *   depending on the "state" variable. Further explained in the "check_rocket" method.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param state - variable that determines if line is drawn full black (1) or dotted (2)
    */
    void draw_rocket1(N5110 &lcd, int state);
    
    
    /** Draw Second Rocket
    *   
    *   Draws the second rocket from the left in the weapons bar onto the LCD. It is drawn full black or dotted
    *   depending on the "state" variable. Further explained in the "check_rocket" method.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param state - variable that determines if line is drawn full black (1) or dotted (2)
    */
    void draw_rocket2(N5110 &lcd, int state);
    
    
    /** Draw Third Rocket
    *   
    *   Draws the third rocket from the left in the weapons bar onto the LCD. It is drawn full black or dotted
    *   depending on the "state" variable. Further explained in the "check_rocket" method.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param state - variable that determines if line is drawn full black (1) or dotted (2)
    */
    void draw_rocket3(N5110 &lcd, int state);
    
    
    /** Draw Star
    *   
    *   Draws the star onto the LCD. It is drawn full black or dotted depending on the "state" variable.
    *   Further explained in the "check_rocket" method.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param state - variable that determines if line is drawn full black (1) or dotted (2)
    */
    void draw_star (N5110 &lcd, int state);
    
    
    /** Check Health (high)
    *   
    *   Draws the health bars onto the LCD depending on how much health the player has (depending on the variable "collisons").
    *   If zero collisions, health bar is full, i.e the heart is filled black. As collisions increase, the health bar
    *   goes down, i.e less of the heart is filled black. This method only deals with the health when it is high.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param collisions - variable that stores how many collisions have been registered
    */
    void check_health_high(N5110 &lcd, int collisions);
    
    
    /** Check Health (low)
    *   
    *   Draws the health bars onto the LCD depending on how much health the player has (depending on the variable "collisons").
    *   If zero collisions, health bar is full, i.e the heart is filled black. As collisions increase, the health bar
    *   goes down, i.e less of the heart is filled black. This method only deals with the health when it is low.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param collisions - variable that stores how many collisions have been registered
    */
    void check_health_low(N5110 &lcd, int collisions);
    
    
    /** Check Rocket
    *   
    *   Draws the rockets in the weapons bar onto the LCD depending on how many the player has (depending on the variable
    *   "ammo"). If full ammo, all 3 rockets are drawn full black, i.e drawn with the state set to 1. After one shot (ammo = 2),
    *   the first two are still drawn full black and the third one is drawn dotted, i.e the first two are drawn with a state of 1,
    *   and the third one is drawn with a state of 2, etc.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param ammo - variable that stores how many rockets are available
    */
    void check_rocket(N5110 &lcd, int ammo);
    
    
    /** Check Star
    *
    *   Draws the star in the weapons bar onto the LCD depending on if the player has one available (depending on the variable
    *   "star"). If "star" is set to true, it is drawn full black, i.e drawn with the state set to 1, if it is set to false, it
    *   is drawn dotted, i.e drawn with the state set to 2.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param star - variable that stores whether a star is available or not
    */
    void check_star(N5110 &lcd, bool star);
    
private:

};

#endif