#ifndef FRIENDLY_H
#define FRIENDLY_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/** Friendly Class
@brief Used for controlling the friendly ship in the Nemesis game. Includes drawing, checking, and updating functions. 
@brief Incorporates N5110.h and Gamepad.h files by Craig A. Evans.

@brief Revision 1.0

@author Musallam M. M. Bseiso
@date   3rd May 2017
*/


class Friendly
{
public:
    
    /// Constructor and destructor:
    Friendly();
    ~Friendly();
    
    
    ////////////////////////////////
    //////// PUBLIC METHODS
    ////////////////////////////////
    
    
    /** Initialize Friendly
    *   
    *   Initializes friendly ship x & y positions.
    */
    void init();
    
    
    /** Draw Friendly
    *   
    *   Draws the friendly ship onto the LCD, in accordance with the parameters initialized in the "init" method.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    */
    void draw(N5110 &lcd);
    
    
    /** Update Friendly (basic)
    *   
    *   Updates the friendly ship's x and y position. X and y positions are altered by adding/subtracting speeds,
    *   which depend on the direction (d) of the analog stick. The speed is defined as the magnitude of the movement
    *   of the analog stick multiplied by an arbitrary number (set as 4). This method only deals with the basic 
    *   horizontal and vertical movements of the friendly ship (North, South, West, East).
    *   @param d - direction of analog stick
    *   @param mag - magnitude of movement of analog stick
    */
    void update_basic(Direction d,float mag);
    
    
    /** Update Friendly (diagonal)
    *   
    *   Updates the friendly ship's x and y position. X and y positions are altered by adding/subtracting speeds,
    *   which depend on the direction (d) of the analog stick. The speed is defined as the magnitude of the movement
    *   of the analog stick multiplied by an arbitrary number (set as 4). This method only deals with the diagonal 
    *   movements of the friendly ship (Northwest, Northeast, Southwest, Southeast).
    *   @param d - direction of analog stick
    *   @param mag - magnitude of movement of analog stick
    */
    void update_diagonal(Direction d,float mag);
    
    
    /** Check Friendly Position
    *
    *   Ensures the friendly ship does not go out of bounds (off screen or into the stats bar) by limiting its x and
    *   y positions.
    */
    void check_pos();
    
    
    /** Get Friendly Position
    *
    *   Obtains the position (x and y coordinates) of the friendly ship at any given time, into a two-dimensional vector.
    *   @returns p - the Vector2D of the parameters
    */
    Vector2D get_pos();

private:
    
    
    ////////////////////////////////
    //////// PRIVATE VARIABLES
    ////////////////////////////////
    
    int _speed;
    
    /// Integer variables to store the x and y coordinates of the friendly ship:
    int _x;
    int _y;

};
#endif