#ifndef ENGINE_H
#define ENGINE_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "Enemy1.h"
#include "Enemy2.h"
#include "Enemy3.h"
#include "Enemy4.h"
#include "Enemy5.h"
#include "Enemy6.h"
#include "Friendly.h"
#include "Rocket.h"
#include "Stats.h"

/** Engine Class
@brief Engine for interfacing with the Nemesis game. Includes drawing, checking, game state-changing methods, 
@brief for Friendly and Enemy ships, Weapons, and Stats bar.
@brief Incorporates N5110.h and Gamepad.h files by Craig A. Evans.

@brief Revision 1.0

@author Musallam M. M. Bseiso
@date   3rd May 2017
*/

class Engine
{

public:

    /// Constructor and destructor:
    Engine();
    ~Engine();
    
    
    ////////////////////////////////
    //////// PUBLIC VARIABLES 
    ////////////////////////////////
    
    
    /// Integer variable that stores the collisions the friendly ship has with the enemy ships:
    int collisions;
    
    /// Integer variable that stores the number of the wave the player is currently in (or the number of waves survived):
    int wave_counter;
    
    /// Boolean variable that stores if the player has a star available or not:
    bool star;
    
    /// Integer variable that stores the number of rockets the player has available:
    int ammo;
    
    /// Integer variable that stores the number of pauses the player has available:
    int pauses;
    
    /// Boolean variable that stores if a rocket has been shot or not:
    bool rocket_shot_trigger;
    
    /// Boolean variable that stores if health has been regenerated or not:
    bool health_regen_trigger;
    
    /// Boolean variable that stores if a rocket has been regenerated or not:
    bool rocket_regen_trigger;
    
    /// Boolean variable that stores if a star has been regenerated or not:
    bool star_regen_trigger;
    
    
    ////////////////////////////////
    //////// PUBLIC METHODS
    ////////////////////////////////
    
    
    /** Initialize Ships
    *   
    *   Initializes friendly and enemy ships' velocity (enemies) and x & y positions (both).
    *   @param speed - velocity of ship
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */
    void init(int speed, N5110 &lcd, Gamepad &pad, int _collisions, int _wave_counter, int _ammo, bool _star, int _pauses);
    
    
    /** Read Input
    *
    *   Reads direction (_d) and magnitude (_mag) from gamepad analog stick.
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */
    void read_input(Gamepad &pad);
    
    
    /** Draw Elements
    *   
    *   Draws grid, health, weapons, wave counter, and friendly and enemy ships onto the LCD
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    */ 
    void draw_all(N5110 &lcd); 
    
    
    /** Check All
    *   
    *   Method that includes all other checking methods
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */
    void check_all(N5110 &lcd, Gamepad &pad);
    
    
    /** Update All
    *
    *   Updates friendly and enemy ship parameters (position), as well as making sure the friendly ship doesn't go off screen
    *   by limiting its x and y positions.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void update_all(N5110 &lcd, Gamepad &pad);
    
    
    /** Shoot Rocket
    *
    *   Triggers a rocket to be shot from the friendly ship, destroying enemy ship opposite, at the press of a button (B).
    *   Checks if rockets ("ammo" variable) available, draws rocket onto screen, triggers "check_enemy_death" method, subtracts 1 from 
    *   "ammo", and outputs a sound. First of two weapon types.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void shoot_rocket(N5110 &lcd, Gamepad &pad);
    
    
    /** Shoot Star
    *
    *   Triggers a star to be shot from the friendly ship, destroying all enemy ships, at the press of a button (A).
    *   Checks if star ("star" variable) available, sets "star" to false, and outputs a sound. Second of two weapon types.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void shoot_star(N5110 &lcd, Gamepad &pad);
    
private:
    
    
    ////////////////////////////////
    //////// PRIVATE VARIABLES
    ////////////////////////////////
    
    
    /// Integer variable that stores the speed of the enemy ships:
    int _speed;
    
    /// Variable that stores the direction of the analog stick:
    Direction _d;
    
    /// Floating point variable that stores the magnitude of the analog stick movement:
    float _mag;
    
    /// Objects created:
    Friendly _friendly;
    Enemy1 _enemy1;
    Enemy2 _enemy2;
    Enemy3 _enemy3;
    Enemy4 _enemy4;
    Enemy5 _enemy5;
    Enemy6 _enemy6;
    Rocket _rocket;
    Stats _stats;
    
    ////////////////////////////////
    //////// PRIVATE METHODS
    ////////////////////////////////
    

    /** Check Enemy Pass (high)
    *
    *   Reinitializes enemy ships that have passed off-screen (x < 0) to their starting position. As Enemy1 passes, 
    *   increments wave counter  (adds 1 to the "wave_counter" variable), then resets rocket and star regeneration 
    *   trigger to false (sets "rocket_regen_trigger" and "star_regen_trigger" to false) to allow further regeneration
    *   at next wave. This method only deals with reinitializing the upper 3 enemies (Enemy1, Enemy2, and Enemy3).
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void check_enemy_pass_high(Gamepad &pad);
    
    
    /** Check Enemy Pass (low)
    *
    *   Reinitializes enemy ships that have passed off-screen (x < 0) to their starting position. This method only deals 
    *   with reinitializing the lower 3 enemies (Enemy4, Enemy5, and Enemy6).
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void check_enemy_pass_low(Gamepad &pad);
    
    
    /** Check Enemy Death (high)
    *
    *   Reinitializes enemy ships that have been shot by a rocket to their starting position. This method only deals with
    *   reinitializing the upper 3 enemies (Enemy1, Enemy2, and Enemy3).
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void check_enemy_death_high(Gamepad &pad);
    
    
    /** Check Enemy Death (low)
    *
    *   Reinitializes enemy ships that have been shot by a rocket to their starting position. This method only deals with
    *   reinitializing the lower 3 enemies (Enemy4, Enemy5, and Enemy6).
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void check_enemy_death_low(Gamepad &pad);
    

    /** Check Friendly Death
    *
    *   Registers friendly collisions with an enemy ship (adds 1 to "collisions" variable).
    */ 
    void check_friendly_death(Gamepad &pad, Vector2D ship_pos);
    
    
    /** Check Friendly Death All
    *
    *   Method that includes "check_friendly_death" method for all 6 enemy ships.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */
    void check_friendly_death_all(N5110 &lcd, Gamepad &pad);
    
    
    /** Check Health Replenish
    *
    *   Regenerates health (subtracts 1 to "collisions" variable) at certain waves, if "health_regen_trigger" is false, and
    *   the player has less than full health already (only executes if "collisions" is non-zero).
    *   Then sets "health_regen_trigger" to true to prevent excessive health regeneration.
    */
    void check_health_replenish();
    
    
    /** Check Rocket Replenish
    *
    *   Regenerates rockets (adds 1 to "ammo" variable) at certain waves, if "rocket_regen_trigger" is false, and
    *   the player has less than 3 rockets already, as the maximum is 3 (only executes if "ammo" is less than 3).
    *   Then sets "rocket_regen_trigger" to true to prevent excessive rocket regeneration.
    */
    void check_rocket_replenish();
    
    
    /** Check Star Replenish
    *
    *   Regenerates star (sets "star" variable to true) at certain waves, if "star_regen_trigger" is false, and
    *   the player does not already have a star, as the maximum is 1 (only executes if "star" is false). Then sets
    *   "star_regen_trigger" to true to prevent excessive star regeneration.
    */ 
    void check_star_replenish();
    
    
    /** Check Level 2
    *
    *   Moves on to level 2 if wave 5 has been reached. A dialog box is drawn onto the LCD, and the player can move on
    *   to the next level by pressing a button (BACK). Reinitializes the friendly ship at it starting position, and the
    *   enemy ships at their starting positions at an elevated speed (at LEVEL_TWO, which is defined as 3). 
    *   Sets the wave to 6 (sets "wave_counter" variable to 6).
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void check_level_two(N5110 &lcd, Gamepad &pad);
    
    
    /** Check Level 3
    *
    *   Moves on to level 3 if wave 25 has been reached. A dialog box is drawn onto the LCD, and the player can move on
    *   to the next level by pressing a button (BACK). Reinitializes the friendly ship at it starting position, and the
    *   enemy ships at their starting positions at an elevated speed (at LEVEL_THREE, which is defined as 4). 
    *   Sets the wave to 26 (sets "wave_counter" variable to 26).
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void check_level_three(N5110 &lcd, Gamepad &pad);
    
    
    /** Check Level 4
    *
    *   Moves on to level 4 if wave 50 has been reached. A dialog box is drawn onto the LCD, and the player can move on
    *   to the next level by pressing a button (BACK). Reinitializes the friendly ship at it starting position, and the
    *   enemy ships at their starting positions at an elevated speed (at LEVEL_FOUR, which is defined as 5). 
    *   Sets the wave to 51 (sets "wave_counter" variable to 51).
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void check_level_four(N5110 &lcd, Gamepad &pad);
    
    
    /** Check Level 5
    *
    *   Moves on to level 5 if wave 100 has been reached. A dialog box is drawn onto the LCD, and the player can move on
    *   to the next level by pressing a button (BACK). Reinitializes the friendly ship at it starting position, and the
    *   enemy ships at their starting positions at an elevated speed (at LEVEL_FIVE, which is defined as 6). 
    *   Sets the wave to 101 (sets "wave_counter" variable to 101).
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */ 
    void check_level_five(N5110 &lcd, Gamepad &pad);
    
    
    /** Check Pause
    *
    *   Pauses the game if a button (START) is pressed, and the player has remaining pauses to use ("pause" variable is > 0).
    *   Flashes the pad's LEDs and switches the LCD's mode between normal and inverse until a button (START) is pressed again.
    *   Once unpaused, removes one pause from the player (subtracts 1 from the "pauses" variable).
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */
    void check_pause(N5110 &lcd, Gamepad &pad);
    
    
    /** Check LCD Mode
    *
    *   Toggles 'night' mode (LCD's inverse mode) if a button (LEFT) is pressed. Sets the mode back to normal (LCD's normal mode)
    *   if a button (RIGHT) is pressed.
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */
    void check_mode_toggle(N5110 &lcd, Gamepad &pad);
    
    
    /** Check for Gameover Sequence
    *
    *   Ends the game if the player has no lives remaining ("collisions" variable is above 5). A dialog box is printed onto the LCD,
    *   showing the player's score (waves survived). Once a button is pressed (BACK), the friendly ship is reinitialized at its 
    *   starting position, and the enemy ships at their starting positions at the initial speed (at LEVEL_ONE, which is defined as 2). 
    *   @param N5110 - nokia LCD library
    *   @param lcd - pointer to nokia LCD library
    *   @param Gamepad - gamepad library
    *   @param pad - pointer to gamepad library
    */
    void check_gameover(N5110 &lcd, Gamepad &pad);
    
};

#endif