#include "mbed.h"
#include "rtos.h"
#include "tinysh.h"

Serial pc(USBTX, USBRX);

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
/////// RTOS TESTING ///////////////////////////////////////////

Mutex stdio_mutex;

void notify(const char* name, int state)
{
    stdio_mutex.lock();
    printf("%s: %d\n\r", name, state);
    stdio_mutex.unlock();
}

Semaphore two_slots(2);

void test_thread(void const *name)
{
    while (true) {
        two_slots.wait();
        printf("%s\n\r", (const char*)name);
        Thread::wait(1000);
        two_slots.release();
    }
}

DigitalOut led(LED1);

void led_thread(void const *argument)
{
    while (true) {
        // Signal flags that are reported as event are automatically cleared.
        Thread::signal_wait(0x1);
        led = !led;
    }
}

void led_send()
{
    Thread thread(led_thread);

    while (true) {
        Thread::wait(1000);
        thread.signal_set(0x1);
    }
}


typedef struct {
    float    voltage;   /* AD result of measured voltage */
    float    current;   /* AD result of measured current */
    uint32_t counter;   /* A counter value               */
} message_t;

MemoryPool<message_t, 16> mpool;
Queue<message_t, 16> queue;

/* Send Thread */
void send_thread (void const *args)
{
    uint32_t i = 0;
    while (true) {
        i++; // fake data update
        message_t *message = mpool.alloc();
        message->voltage = (i * 0.1) * 33;
        message->current = (i * 0.1) * 11;
        message->counter = i;
        queue.put(message);
        Thread::wait(1000);
    }
}

int main3 (void)
{
    Thread thread(send_thread);

    while (true) {
        osEvent evt = queue.get();
        if (evt.status == osEventMessage) {
            message_t *message = (message_t*)evt.value.p;
            printf("\nVoltage: %.2f V\n\r"   , message->voltage);
            printf("Current: %.2f A\n\r"     , message->current);
            printf("Number of cycles: %u\n\r", message->counter);

            mpool.free(message);
        }
    }
}





/* Mail */
typedef struct {
    float    voltage; /* AD result of measured voltage */
    float    current; /* AD result of measured current */
    uint32_t counter; /* A counter value               */
} mail_t;

Mail<mail_t, 16> mail_box;

void send_thread2 (void const *args)
{
    uint32_t i = 0;
    while (true) {
        i++; // fake data update
        mail_t *mail = mail_box.alloc();
        mail->voltage = (i * 0.1) * 33;
        mail->current = (i * 0.1) * 11;
        mail->counter = i;
        mail_box.put(mail);
        Thread::wait(1000);
    }
}

int main4 (void)
{
    Thread thread(send_thread2);

    while (true) {
        osEvent evt = mail_box.get();
        if (evt.status == osEventMail) {
            mail_t *mail = (mail_t*)evt.value.p;
            printf("\nVoltage: %.2f V\n\r"   , mail->voltage);
            printf("Current: %.2f A\n\r"     , mail->current);
            printf("Number of cycles: %u\n\r", mail->counter);

            mail_box.free(mail);
        }
    }
}

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
/////// HARDWARE TEST KL25 ACCEL       ///////////////////////////


#include "MMA8451Q.h"

#if   defined (TARGET_KL25Z) || defined (TARGET_KL46Z)
PinName const SDA = PTE25;
PinName const SCL = PTE24;
#elif defined (TARGET_KL05Z)
PinName const SDA = PTB4;
PinName const SCL = PTB3;
#else
#error TARGET NOT DEFINED
#endif

#define MMA8451_I2C_ADDRESS (0x1d<<1)

PwmOut rled(LED1);
PwmOut gled(LED2);
PwmOut bled(LED3);

MMA8451Q acc(SDA, SCL, MMA8451_I2C_ADDRESS);

void kl25z_read_accel(int argc, char **argv)
{
    float x, y, z;

    x = rled = 1.0 - abs(acc.getAccX());
    y = gled = 1.0 - abs(acc.getAccY());
    z = bled = 1.0 - abs(acc.getAccZ());
    pc.printf("X: %1.2f, Y: %1.2f, Z: %1.2f\r\n", x, y, z);
}

tinysh_cmd_t cmd_accel= {0,"accel","read ACCEL accel","[args]",kl25z_read_accel,0,0,0};

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
/////// HARDWARE TEST KL25 TSI       ///////////////////////////


#include "tsi_sensor.h"
/* This defines will be replaced by PinNames soon */
#if defined (TARGET_KL25Z) || defined (TARGET_KL46Z)
#define ELEC0 9
#define ELEC1 10
#elif defined (TARGET_KL05Z)
#define ELEC0 9
#define ELEC1 8
#else
#error TARGET NOT DEFINED
#endif

TSIAnalogSlider tsi(ELEC0, ELEC1, 40);

void kl25z_read_tsi(int argc, char **argv)
{
    pc.printf("tsi=%f\r\n",tsi.readPercentage());
}

tinysh_cmd_t cmd_tsi= {0,"tsi","read TSI sensor value","[args]",kl25z_read_tsi,0,0,0};

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
/////// TINY SHELL EXAMPLE AS THREAD ///////////////////////////



void foo_fnt(int argc, char **argv)
{
    printf("foo command called\r\n");
    for(int i=0; i<argc; i++) {
        printf("argv[%d]=\"%s\"\r\n",i,argv[i]);
    }
}
tinysh_cmd_t myfoocmd= {0,"foo","foo command","[args]",foo_fnt,0,0,0};






void tinysh_char_out(unsigned char c)
{
    pc.putc(c);
}

void tinyshell_thread(void const *args)
{
    pc.baud(115200);
    pc.printf("kl25z tiny shell build %s %s\r\n",__DATE__,__TIME__);
    tinysh_set_prompt("$ ");
    tinysh_add_command(&myfoocmd);
    tinysh_add_command(&cmd_tsi);
    tinysh_add_command(&cmd_accel);




    while(true) {
        tinysh_char_in( pc.getc() );
    }

}

////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////










int main()
{
    
    //kl25z led power is inverted 1=off 0=on.
    rled=0.99;
    gled=0.99;
    bled=0.99;

    //
    Thread thread_shell(tinyshell_thread);

    Thread::wait(osWaitForever);
}
