/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <stdarg.h>
#include "mbed.h"
#include "UbloxATCellularInterface.h"
#include "INA219.hpp"

// You must select the correct interface library for your board, by
// uncommenting the correct line below. Supported combinations are
// indicated with a "Y" in the table below.
//
//                            C030_U201   C030_N211      C027     C030_R41XM
// UbloxATCellularInterface       Y            -           Y          Y
// UbloxATCellularInterfaceN2xx   -            Y           -          -
// Note: the N211 module supports only UDP, not TCP

// UbloxATCellularInterface and UbloxATCellularInterfaceN2xx
// uses an IP stack on the cellular module and hence uses less RAM (significant on C027).
// This also allows other AT command operations (e.g. sending an SMS) to happen
// during a data transfer (for which you should replace the
// UbloxATCellularInterface library with the UbloxATCellularInterfaceExt
// library).  However, it is slower than using the LWIP/PPP on the mbed
// MCU interface since more string parsing is required.
#define INTERFACE_CLASS  UbloxATCellularInterface
//#define INTERFACE_CLASS  UbloxATCellularInterfaceN2xx

// The credentials of the SIM in the board.  If PIN checking is enabled
// for your SIM card you must set this to the required PIN.
#define PIN "0000"

// Network credentials.  You should set this according to your
// network/SIM card.  For C030 non-N2xx boards, leave the parameters as NULL
// otherwise, if you do not know the APN for your network, you may
// either try the fairly common "internet" for the APN (and leave the
// username and password NULL), or you may leave all three as NULL and then
// a lookup will be attempted for a small number of known networks
// (see APN_db.h in mbed-os/features/netsocket/cellular/utils).
#define APN         "default"
#define USERNAME    NULL
#define PASSWORD    NULL

// Uncomment the following line to enable Icellular Current measurement.
// Current drawn by modem is printed on serial every 2 seconds.
#define CURRENT_MEASUREMENT

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

// The user button
volatile bool buttonPressed = false;
#ifdef TARGET_UBLOX_C030_R412M
volatile bool modem_asleep = false;
#ifdef CURRENT_MEASUREMENT
AnalogIn ain_icellular(MDMCURRENTSENSE);
Thread icell_thread;
#endif
#endif

Thread *cellularThread;

RawSerial  pc(PD_5, PD_6);
RawSerial  dev(D1, D0);

INA219 ina219(I2C_SDA, I2C_SCL, 0x40, 400000, RES_10BITS);

Ticker measure;
float refresh_rate = 1000;

static rtos::Mutex trace_mutex;

static void good() {
    ledGreen = 0;
    ledBlue = 1;
    ledRed = 1;
}

static void bad() {
    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;
}

static void event() {
    ledBlue = 0;
    ledRed = 1;
    ledGreen = 1;
}

static void pulseEvent() {
    event();
    wait_us(500000);
    good();
}

static void ledOff() {
    ledBlue = 1;
    ledRed = 1;
    ledGreen = 1;
}

static void print_function(const char *format, ...)
{
    trace_mutex.lock();
    va_list arglist;
    va_start( arglist, format );
    vprintf(format, arglist);
    va_end( arglist );
    trace_mutex.unlock();
}

static void printNtpTime(char * buf, int len)
{
    time_t timestamp = 0;
    struct tm *localTime;
    char timeString[25];
    time_t TIME1970 = 2208988800U;

    if (len >= 43) {
        timestamp |= ((int) *(buf + 40)) << 24;
        timestamp |= ((int) *(buf + 41)) << 16;
        timestamp |= ((int) *(buf + 42)) << 8;
        timestamp |= ((int) *(buf + 43));
        timestamp -= TIME1970;
        localTime = localtime(&timestamp);
        if (localTime) {
            if (strftime(timeString, sizeof(timeString), "%a %b %d %H:%M:%S %Y", localTime) > 0) {
                print_function("NTP timestamp is %s.\n", timeString);
            }
        }
    }
}

static void cbButton()
{
    buttonPressed = true;
}

void pc_recv()
{
    while(pc.readable()) {
        dev.putc(pc.getc());
    }
}
void dev_recv()
{
    while(dev.readable()) {
        pc.putc(dev.getc());
    }
}

void read_current()
{
    cellularThread->signal_set(0x01);
}

void show_current()
{
    while(1) {
        Thread::signal_wait(0x01);
        print_function("Current draw: %f mA\r\n", ina219.read_current_mA());
    }
}


void init_modem(INTERFACE_CLASS *interface) {
    int x;

    for (x = 0; interface->connect(PIN) != 0; x++) {
        if (x > 0) {
            bad();
            print_function("Retrying (have you checked that an antenna is plugged in and your APN is correct?)...\n");
        }
    }
}
#ifdef TARGET_UBLOX_C030_R412M
void psm_going_in_cb(void *param)
{
    print_function("PSM callback function:: Modem going in to sleep\n");
    modem_asleep = true;
}
#ifdef CURRENT_MEASUREMENT
float calculate_icellular_samples() {
    float ain=0.0f;
    float icellular_val;
    const int c_number_of_analog_samples = 50;

    ain = 0;
    for(int i = 0; i < c_number_of_analog_samples; i++) {
        ain = (ain + ain_icellular.read());
        ThisThread::sleep_for(20);
    }
    ain = ain/c_number_of_analog_samples;
    icellular_val = (ain*1.8*1000)/7.0f;


    print_function("Voltage in mV: %f\n", icellular_val * 7.0f);
    print_function("Current draw in mA: %f\n\n", icellular_val);

    return icellular_val;
}

void icell_thread_handler() {

    while(1) {
        calculate_icellular_samples();
    }
}
#endif
#endif

/* This example program for the u-blox C030-R410M board instantiates
 * the UbloxATCellularInterface and uses it to make a simple sockets
 * connection to a server, using 2.pool.ntp.org for UDP and
 * developer.mbed.org for TCP. It also showcases the 3GPP PSM feature.
 * For a more comprehensive example, where higher layer protocols
 * make use of the same sockets interface, see example-ublox-mbed-client.
 * Progress may be monitored with a serial terminal running at 9600 baud.
 * The LED on the C030 board will turn green when this program is
 * operating correctly, pulse blue when a sockets operation is completed
 * and turn red if there is a failure.
 */
int main()
{
#ifdef TARGET_UBLOX_C030_R412M
/*#ifdef CURRENT_MEASUREMENT
    //current monitoring using Icellular
    icell_thread.start(icell_thread_handler);
#endif*/
    int status = 0, pt = 0, at = 0;
#endif

    cellularThread = new Thread(osPriorityNormal, OS_STACK_SIZE, NULL, "cellular_thread");
    cellularThread->start(callback(show_current));

    float refresh_interval = refresh_rate;
    measure.attach(&read_current, 1.0f);

    //INTERFACE_CLASS *interface = new INTERFACE_CLASS();
    // If you need to debug the cellular interface, comment out the
    // instantiation above and uncomment the one below.
    // For the N2xx interface, change xxx to MBED_CONF_UBLOX_CELL_BAUD_RATE,
    // while for the non-N2xx interface change it to MBED_CONF_UBLOX_CELL_N2XX_BAUD_RATE.
    INTERFACE_CLASS *interface = new INTERFACE_CLASS(D1, D0,
                                                     115200,
                                                     true);

    UDPSocket sockUdp;
    SocketAddress udpServer;
    SocketAddress udpSenderAddress;
    char buf[1024];
    int x;
    int upsv_status;
#ifdef TARGET_UBLOX_C027
    // No user button on C027
    InterruptIn userButton(NC);
#else
    InterruptIn userButton(SW0);
#endif

    // Attach a function to the user button
    userButton.rise(&cbButton);

    good();

    print_function("Initializing modem, please wait.\n");
    if (interface->init(PIN) == false) //setup modem
    {
    	print_function("Initialization complete.\n");
    }
    pulseEvent();

    pc.baud(115200);
    dev.baud(115200);

    pc.attach(&pc_recv, Serial::RxIrq);
    dev.attach(&dev_recv, Serial::RxIrq);

    //print_function("sleeping 30 seconds \n");
    //wait_us(30000000);


#ifdef TARGET_UBLOX_C030_R412M

    if (interface->get_idle_mode(&upsv_status)) {
        printf("\n\nIdle mode is %s\n", upsv_status ? "Enabled" : "Disabled");
    }

//    if (!upsv_status) {
//        printf("Enabling idle mode...\n");
//        if (interface->set_idle_mode(true) == false)
//        {
//            printf("Unable to set idle mode, is PSM enabled?\n");
//        }
//        if (interface->get_idle_mode(&upsv_status)) {
//            printf("Idle mode is %s\n", upsv_status ? "Enabled" : "Disabled");
//        }
//    }

    print_function("Enable PSM in 10...\n");
    if (1){//interface->set_power_saving_mode(0, 0)) { //enable PSM
        print_function("PSM enabled. Attaching CB function and rebooting the module\n");
        interface->attach_cb_psm_going_in(&psm_going_in_cb, NULL); //register callback

        //reset modem so that PSM settings can take effect
        //interface->reboot_modem();
        //wait_us(5000000); //give modem a little time

        print_function("please wait up to 180 seconds for network registration to complete...\n");
        //try to re-init modem and perform registration
        interface->set_credentials(APN, USERNAME, PASSWORD);
        interface->connect(PIN);
        /*for (x = 0; interface->connect(PIN) != 0; x++) {
            if (x > 0) {
                bad();
                print_function("Retrying (have you checked that an antenna is plugged in and your APN is correct?)...\n");
            }
        }*/

        interface->get_power_saving_mode(&status, &pt, &at); //read assigned values
        print_function("PSM status: %s\nAssigned Periodic TAU: %d\nAssigned Active time: %d\n", status ? "enabled" : "disabled", pt, at);
        pulseEvent();
    }

    wait_us(60000000);

    interface->set_idle_mode(true);

    wait_us(90000000); //give modem a little time

    //sometimes modem goes in to PSM before we can do any UDP/TCP transfers
    if (modem_asleep == true) {
        print_function("Modem is in PSM, waking up and initializing it\n");
        interface->wakeup_modem(); //this wakes up the modem and also CellularBase gets synced with modem state.
        init_modem(interface);
        modem_asleep = false;
        print_function("Initialization complete\n");
    }

    print_function("Re-Initializing modem, please wait.\n");
    if (interface->init(PIN) == false) //setup modem
    {
    	print_function("Initialization complete.\n");
    }

    while (!buttonPressed) {
        pulseEvent();

        wait_us(500000);
    }

#endif

    /*print_function("\nGetting the IP address of \"2.pool.ntp.org\"...\n");
    if ((interface->gethostbyname("2.pool.ntp.org", &udpServer) == 0)) {
        pulseEvent();

        udpServer.set_port(123);
        print_function("\"2.pool.ntp.org\" address: %s on port %d.\n", udpServer.get_ip_address(), udpServer.get_port());

        print_function("Performing socket operations in a loop (until the user button is pressed on C030 or forever on C027)...\n");
        while (!buttonPressed) {
#ifdef TARGET_UBLOX_C030_R412M
            if (modem_asleep == true) {
                print_function("Modem is in PSM, waking up and initializing it\n");
                interface->wakeup_modem(); //this wakes up the modem and also CellularBase gets synced with modem state.
                init_modem(interface);
                modem_asleep = false;
                print_function("Initialization complete\n");
                wait_ms(5000);
            } else {
#endif
                print_function("Opening a UDP socket...\n");
                if ((sockUdp.open(interface)) == 0) {
                    // UDP Sockets
                    pulseEvent();
                    print_function("UDP socket open.\n");
                    sockUdp.set_timeout(20000);
                    print_function("Sending time request to \"2.pool.ntp.org\" over UDP socket...\n");
                    memset (buf, 0, sizeof(buf));
                    *buf = '\x1b';
                    if (sockUdp.sendto(udpServer, (void *) buf, 48) == 48) {
                        pulseEvent();
                        print_function("Socket send completed, waiting for UDP response...\n");
                        x = sockUdp.recvfrom(&udpSenderAddress, buf, sizeof (buf));
                        if (x > 0) {
                            pulseEvent();
                            print_function("Received %d byte response from server %s on UDP socket:\n"
                                "-------------------------------------------------------\n",
                                x, udpSenderAddress.get_ip_address());
                            printNtpTime(buf, x);
                            print_function("-------------------------------------------------------\n");
                        }
                    }
                    print_function("Closing socket...\n");
                    sockUdp.close();
                    pulseEvent();
                    print_function("Socket closed.\n");
                }
#ifdef TARGET_UBLOX_C030_R412M
                while(modem_asleep == false) { //modem is awake, let it go to sleep again
                    print_function("Waiting for modem to go to PSM sleep\n");
                    wait_ms(5000);
                }
            }
#endif // TARGET_UBLOX_C030_R412M

            wait_ms(20000);
#ifndef TARGET_UBLOX_C027
            print_function("\n\n[Checking if user button has been pressed]\n");
#endif
        }

        pulseEvent();
        print_function("User button was pressed, stopping...\n");
        interface->disconnect();
        ledOff();
        print_function("Stopped.\n");
    } else {
        bad();
        print_function("Unable to get IP address of \"2.pool.ntp.org\".\n");
    }*/
}

// End Of File
