#include "LCD.hpp"
#include "rtos.h"

#define GRAPHICS_FORMAT (DisplayBase::GRAPHICS_FORMAT_RGB565)
#define WR_RD_WRSWA (DisplayBase::WR_RD_WRSWA_32_16BIT)

volatile int32_t LCD::vsync_count = 0;
uint8_t LCD::user_frame_buffer1[];
uint8_t LCD::user_frame_buffer2[];
frame_buffer_t LCD::frame_buffer_info;
LCD LCD::instance;

void LCD::IntCallbackFunc_LoVsync(DisplayBase::int_type_t MBED_UNUSED int_type) {
    /* Interrupt callback function for Vsync interruption */
    if (vsync_count > 0) {
        vsync_count--;
    }
}

LCD::LCD() : lcd_pwon(P7_15), lcd_blon(P8_1), lcd_cntrst(P8_15)
{
}

void LCD::start()
{
    DisplayBase::graphics_error_t error;
    DisplayBase::lcd_config_t lcd_config;
    PinName lvds_pin[8] = {
        /* data pin */
        P5_7, P5_6, P5_5, P5_4, P5_3, P5_2, P5_1, P5_0
    };
    
    lcd_pwon = 0;
    lcd_blon = 0;
    rtos::Thread::wait(100);
    lcd_pwon = 1;
    lcd_blon = 1;
    
    Display.Graphics_Lvds_Port_Init(lvds_pin, 8);
    
    /* Graphics initialization process */
    lcd_config = LcdCfgTbl_LCD_shield;
    error = Display.Graphics_init(&lcd_config);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        mbed_die();
    }
    
    /* Interrupt callback function setting (Vsync signal output from scaler 0) */
    error = Display.Graphics_Irq_Handler_Set(DisplayBase::INT_TYPE_S0_LO_VSYNC, 0, LCD::IntCallbackFunc_LoVsync);
    if (error != DisplayBase::GRAPHICS_OK) {
        printf("Line %d, error %d\n", __LINE__, error);
        mbed_die();
    }
    
    memset(user_frame_buffer1, 0, sizeof(user_frame_buffer1));
    memset(user_frame_buffer2, 0, sizeof(user_frame_buffer2));
    frame_buffer_info.buffer_address[0] = user_frame_buffer1;
    frame_buffer_info.buffer_address[1] = user_frame_buffer2;
    frame_buffer_info.buffer_count      = 2;
    frame_buffer_info.show_buffer_index = 0;
    frame_buffer_info.draw_buffer_index = 0;
    frame_buffer_info.width             = LCD_PIXEL_WIDTH;
    frame_buffer_info.byte_per_pixel    = FRAME_BUFFER_BYTE_PER_PIXEL;
    frame_buffer_info.stride            = LCD_PIXEL_WIDTH * FRAME_BUFFER_BYTE_PER_PIXEL;
    frame_buffer_info.height            = LCD_PIXEL_HEIGHT;
    frame_buffer_info.pixel_format      = PIXEL_FORMAT_RGB565;
    
    DisplayBase::rect_t rect;
    
    rect.vs = 0;
    rect.vw = LCD_PIXEL_HEIGHT;
    rect.hs = 0;
    rect.hw = LCD_PIXEL_WIDTH;
    Display.Graphics_Read_Setting(DisplayBase::GRAPHICS_LAYER_0, (void *)frame_buffer_info.buffer_address[0],
                                  FRAME_BUFFER_STRIDE, GRAPHICS_FORMAT, WR_RD_WRSWA, &rect);
    Display.Graphics_Start(DisplayBase::GRAPHICS_LAYER_0);
    
    rtos::Thread::wait(200);
    lcd_cntrst.period_us(491);
    lcd_cntrst = 0.1;
    
    errnum_t err;
    Canvas2D_ContextConfigClass config;
    config.frame_buffer = &frame_buffer_info;
    canvas2d = R_RGA_New_Canvas2D_ContextClass(config);
    err = R_OSPL_GetErrNum();
    if (err != 0) {
        printf("Line %d, error %d\n", __LINE__, err);
        while (1);
    }
    
    graphics = canvas2d.c_LanguageContext;
}

void LCD::stop()
{
    lcd_pwon = 0;
    lcd_blon = 0;
}

void LCD::Wait_Vsync(const int32_t wait_count)
{
    /* Wait for the specified number of times Vsync occurs */
    vsync_count = wait_count;
    while (vsync_count > 0) {
        /* Do nothing */
    }
}

void LCD::Swap_FrameBuffer()
{
    if (frame_buffer_info.draw_buffer_index == 1) {
        frame_buffer_info.draw_buffer_index = 0;
    } else {
        frame_buffer_info.draw_buffer_index = 1;
    }
}

void LCD::Update_LCD_Display()
{
    Display.Graphics_Read_Change(DisplayBase::GRAPHICS_LAYER_0,
                                 (void *)frame_buffer_info.buffer_address[frame_buffer_info.draw_buffer_index]);
    Wait_Vsync(1);
}

