#include "mbed.h"

SPI my_spi(p5, p6, p7);
Serial pc(USBTX,USBRX);

#define TNF 0x02
#define TFE 0x01
#define RNE 0x04

#define WREN 0x06
#define WRITE 0x02
#define READ 0x03

/*** My method to write to FeRAM ***/
void spi_write (unsigned char data) {
    // First don't write to the FIFO buffer if it is full
    while (!(LPC_SSP1->SR & TNF))       // While TNF-Bit = 0 (FIFO full)...
        ;                               // Wait
    LPC_SSP1->DR = data;                // Write to FIFO buffer
}

/*** Write a byte to FeRAM ***/
void write_byte (int address, unsigned char data) {
    spi_write(WREN);
    while (!(LPC_SSP1->SR & TFE))       // While TFE-Bit = 1 (FIFO empty)...
        ;                               // Wait
    spi_write(WRITE);                   // Send write command
    spi_write(address >> 16);           // Send top address byte to write to
    spi_write((address >> 8) & 0xFF);   // Send Middle address byte to write to
    spi_write(address & 0xFF);          // Send Bottom address byte to write to
    spi_write(data);                    // Send data to be write
    // Now I need to check that the FIFO transmit buffer is empty exiting the method
    while (!(LPC_SSP1->SR & TFE))       // While TFE-Bit = 0 (FIFO not empty)...
        ;                               // Wait
}

/*** Read a byte from FeRAM ***/
unsigned char read_byte (int address) {
    unsigned char my_val;               // Variable to store the read data
    spi_write(READ);                    // Send read command
    spi_write(address >> 16);           // Send top address byte to read from
    spi_write((address >> 8) & 0xFF);   // Send Middle address byte to read from
    spi_write(address & 0xFF);          // Send Bottom address byte to read from
    // Now the buffer is empty send out a dummy byte and read the buffer
    spi_write(0x00);                    // Send the dummy byte
    // Now I need to empty the FIFO receive buffer...
    while (LPC_SSP1->SR & RNE)          // While RNE-Bit = 1 (FIFO receive buffer not empty)...
        my_val = LPC_SSP1->DR;          // Read the byte in the buffer
    return (my_val);                    // Return the last byte read
}

int main() {
    unsigned char my_val[19200];

    // Set up the SPI port...
    my_spi.frequency(32000000);         // 32MHz is the fastest mbed frequency supported by the FeRAM
    my_spi.format(8,3);                 // Set for mode 3
    LPC_PINCON->PINSEL0 |= 0x00002000;  // Set up SSEL1

    // To start with I need the chip select high and wait 1ms after powering up
    wait(0.001);                        // Must wait 1ms after power on before using FeRAM

    // Create timers
    Timer write_timer, read_timer;

    int val = 88;
    
    write_timer.start();
    // Write the bytes...
    for (int i = 0; i < 19200; i++) {
        write_byte (i, val);
    }
    write_timer.stop();

    read_timer.start();
    // Read the bytes...
    for (int i = 0; i < 19200; i++) {
        my_val[i] = read_byte (i);
    }
    read_timer.stop();

    // Print the data read on screen
    for (int i = 0; i < 19200; i++) {
        if(my_val[i] != val)
            pc.printf("\n\r Error at element %d [%d]",i,my_val[i]);
    }
    
    pc.printf("\n\n\r Write Time = %f ms\n\r Read Time = %f ms",write_timer.read()*1000,read_timer.read()*1000);
}