#include "mbed.h"
#include "FeRAM.h"

FeRAM::FeRAM(PinName mosi, PinName miso, PinName sclk) : _spi(mosi, miso, sclk) {

    _miso = miso;                       // To see which port was used

    // Set up the SPI port...
    _spi.frequency(32000000);           // 32MHz is the fastest mbed frequency supported by the FeRAM
    _spi.format(8,3);                   // Set for mode 3
    if  (_miso == p6) {
        LPC_PINCON->PINSEL0 |= p6_SSEL; // Set up SSEL1
    } else if (_miso == p12) {
        LPC_PINCON->PINSEL1 |= p14_SSEL;// Set up SSEL1
    }

    // To start with I need the chip select high and wait 1ms after powering up
    wait(0.001);                        // Must wait 1ms after power on before using FeRAM
}

/*** My method to write to FeRAM ***/
void FeRAM::spi_write_SSP1 (unsigned char data) {
    // First don't write to the FIFO buffer if it is full
    while (!(LPC_SSP1->SR & TNF))       // While TNF-Bit = 0 (FIFO full)...
        ;                               // Wait
    LPC_SSP1->DR = data;                // Write to FIFO buffer
}

void FeRAM::spi_write_SSP0 (unsigned char data) {
    // First don't write to the FIFO buffer if it is full
    while (!(LPC_SSP0->SR & TNF))       // While TNF-Bit = 0 (FIFO full)...
        ;                               // Wait
    LPC_SSP0->DR = data;                // Write to FIFO buffer
}

/*** Write a byte to FeRAM ***/
void FeRAM::write_byte (int address, unsigned char data) {
    if  (_miso == p6) {
        spi_write_SSP1(WREN);                   // Send the write enable command
        while (!(LPC_SSP1->SR & TFE))           // While TFE-Bit = 1 (FIFO empty)...
            ;                                   // Wait
        spi_write_SSP1(WRITE);                  // Send write command
        spi_write_SSP1(address >> 16);          // Send top address byte to write to
        spi_write_SSP1((address >> 8) & 0xFF);  // Send Middle address byte to write to
        spi_write_SSP1(address & 0xFF);         // Send Bottom address byte to write to
        spi_write_SSP1(data);                   // Send data to be writen
        // Now I need to check that the FIFO transmit buffer is empty exiting the method
        while (!(LPC_SSP1->SR & TFE))           // While TFE-Bit = 0 (FIFO not empty)...
            ;                                   // Wait
    } else if  (_miso == p12) {
        spi_write_SSP0(WREN);                   // Send the write enable command
        while (!(LPC_SSP0->SR & TFE))           // While TFE-Bit = 1 (FIFO empty)...
            ;                                   // Wait
        spi_write_SSP0(WRITE);                  // Send write command
        spi_write_SSP0(address >> 16);          // Send top address byte to write to
        spi_write_SSP0((address >> 8) & 0xFF);  // Send Middle address byte to write to
        spi_write_SSP0(address & 0xFF);         // Send Bottom address byte to write to
        spi_write_SSP0(data);                   // Send data to be writen
        // Now I need to check that the FIFO transmit buffer is empty exiting the method
        while (!(LPC_SSP0->SR & TFE))           // While TFE-Bit = 0 (FIFO not empty)...
            ;                                   // Wait
    }
}

/*** Read a byte from FeRAM ***/
unsigned char FeRAM::read_byte (int address) {
    unsigned char my_val;                       // Variable to store the read data
    if  (_miso == p6) {
        spi_write_SSP1(READ);                   // Send read command
        spi_write_SSP1(address >> 16);          // Send top address byte to read from
        spi_write_SSP1((address >> 8) & 0xFF);  // Send Middle address byte to read from
        spi_write_SSP1(address & 0xFF);         // Send Bottom address byte to read from
        // Now the buffer is empty send out a dummy byte and read the buffer
        spi_write_SSP1(0x00);                   // Send the dummy byte
        // Now I need to empty the FIFO receive buffer...
        while (LPC_SSP1->SR & RNE)              // While RNE-Bit = 1 (FIFO receive buffer not empty)...
            my_val = LPC_SSP1->DR;              // Read the byte in the buffer
    } else if  (_miso == p12) {
        spi_write_SSP0(READ);                   // Send read command
        spi_write_SSP0(address >> 16);          // Send top address byte to read from
        spi_write_SSP0((address >> 8) & 0xFF);  // Send Middle address byte to read from
        spi_write_SSP0(address & 0xFF);         // Send Bottom address byte to read from
        // Now the buffer is empty send out a dummy byte and read the buffer
        spi_write_SSP0(0x00);                   // Send the dummy byte
        // Now I need to empty the FIFO receive buffer...
        while (LPC_SSP0->SR & RNE)              // While RNE-Bit = 1 (FIFO receive buffer not empty)...
            my_val = LPC_SSP0->DR;              // Read the byte in the buffer
    }
    return (my_val);                        // Return the last byte read
}

/*** Write a an array of bytes to FeRAM ***/
void FeRAM::write_multiple_bytes (int start_address, unsigned char* data, int length) {
    if  (_miso == p6) {
        spi_write_SSP1(WREN);                       // Send the write enable command
        while (!(LPC_SSP1->SR & TFE))               // While TFE-Bit = 1 (FIFO empty)...
            ;                                       // Wait
        spi_write_SSP1(WRITE);                      // Send write command
        spi_write_SSP1(start_address >> 16);        // Send top address byte to write to
        spi_write_SSP1((start_address >>8)&0xFF);   // Send Middle address byte to write to
        spi_write_SSP1(start_address & 0xFF);       // Send Bottom address byte to write to
        // Now I can start writing the data
        for (int i = 0; i < length; i++) {
            spi_write_SSP1(data[i]);                // Send the next byte to be writen
        }
        // Now I need to check that the FIFO transmit buffer is empty exiting the method
        while (!(LPC_SSP1->SR & TFE))               // While TFE-Bit = 0 (FIFO not empty)...
            ;                                       // Wait
    } else if  (_miso == p12) {
        spi_write_SSP0(WREN);                       // Send the write enable command
        while (!(LPC_SSP0->SR & TFE))               // While TFE-Bit = 1 (FIFO empty)...
            ;                                       // Wait
        spi_write_SSP0(WRITE);                      // Send write command
        spi_write_SSP0(start_address >> 16);        // Send top address byte to write to
        spi_write_SSP0((start_address >>8)&0xFF);   // Send Middle address byte to write to
        spi_write_SSP0(start_address & 0xFF);       // Send Bottom address byte to write to
        // Now I can start writing the data
        for (int i = 0; i < length; i++) {
            spi_write_SSP0(data[i]);                // Send the next byte to be writen
        }
        // Now I need to check that the FIFO transmit buffer is empty exiting the method
        while (!(LPC_SSP0->SR & TFE))               // While TFE-Bit = 0 (FIFO not empty)...
            ;                                       // Wait
    }
}

/*** Read multiple bytes from FeRAM and store in an array***/
void FeRAM::read_multiple_bytes (int start_address, unsigned char* data, int length) {
    if  (_miso == p6) {
        // First configure the SSEL to cs and pull low to start read
        LPC_PINCON->PINSEL0 &= p6_GPIO;             // Clear bit to set up p0.6 as GPIO
        LPC_GPIO0->FIODIR |= (1 << 6);              // Config as output
        LPC_GPIO0->FIOCLR = (1 << 6);               // Set bit to set output logic low
        // Now pin is set read the data
        spi_write_SSP1(READ);                       // Send read command
        spi_write_SSP1(start_address >> 16);        // Send top address byte to read from
        spi_write_SSP1((start_address >>8)&0xFF);   // Send Middle address byte to read from
        spi_write_SSP1(start_address & 0xFF);       // Send Bottom address byte to read from
        spi_write_SSP1(0x00);                       // Write dummy byte
        // Now I can start reading the data
        for (int i = 0; i < length; i++) {
            spi_write_SSP1(0x00);                   // Write dummy byte
            while (LPC_SSP1->SR & RNE)              // While RNE-Bit = 1 (FIFO receive buffer not empty)...
                data[i] = LPC_SSP1->DR;             // Read the byte in the buffer
        }
        LPC_GPIO0->FIOSET = (1 << 6);               // Set bit to set output logic high
        LPC_PINCON->PINSEL0 |= p6_SSEL;             // Clear bit to set up p0.6 as GPIO
    } else if  (_miso == p12) {
        LPC_PINCON->PINSEL1 &= p14_GPIO;            // Clear bit to set up p0.16 as GPIO
        LPC_GPIO0->FIODIR |= (1 << 16);             // Config as output
        LPC_GPIO0->FIOCLR = (1 << 16);              // Set bit to set output logic low
        // Now pin is set read the data
        spi_write_SSP0(READ);                       // Send read command
        spi_write_SSP0(start_address >> 16);        // Send top address byte to read from
        spi_write_SSP0((start_address >>8)&0xFF);   // Send Middle address byte to read from
        spi_write_SSP0(start_address & 0xFF);       // Send Bottom address byte to read from
        spi_write_SSP0(0x00);                       // Write dummy byte
        // Now I can start reading the data
        for (int i = 0; i < length; i++) {
            spi_write_SSP0(0x00);                   // Write dummy byte
            while (LPC_SSP0->SR & RNE)              // While RNE-Bit = 1 (FIFO receive buffer not empty)...
                data[i] = LPC_SSP0->DR;             // Read the byte in the buffer
        }
        LPC_GPIO0->FIOSET = (1 << 16);              // Set bit to set output logic high
        LPC_PINCON->PINSEL1 |= p14_SSEL;            // Clear bit to set up p0.16 as GPIO
    }
    data[0] = read_byte (start_address);
    data[1] = read_byte (start_address+1);
}