/* Copyright (c) 2012 Martin Smith, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
 * and associated documentation files (the "Software"), to deal in the Software without restriction, 
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, 
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or 
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef MBED_AX12_H
#define MBED_AX12_H

#include "mbed.h"

#define AX12_WRITE_DEBUG 0
#define AX12_READ_DEBUG 0
#define AX12_TRIGGER_DEBUG 0
#define AX12_DEBUG 0

#define AX12_REG_GOAL_POSITION 0x1E
#define AX12_REG_MOVING_SPEED 0x20
#define AX12_REG_POSITION 0x24

#define TRANSMIT 1
#define RECEIVE 0

/** mbed AX-12+ Servo Library - External hardware version
 *
 * Example:
 * @code
 * // Move a servo back and forth...
 * #include "mbed.h"
 * #include "Servo.h"
 * 
 * Serial ax12_bus(p28,p27);
 * AX12 my_ax12(ax12_bus, p29, 1);
 * 
 * int main() {
 *     while(1) {
 *         my_ax12.SetGoal(0);      // move to 0
 *         wait(2);
 *         my_ax12.SetGoal(300);    // move to 300
 *         wait(2);
 *     }
 * }
 * @endcode
 */

class AX12
{

public:
    /** define which pins are used, and the ID of this instance
    */
    AX12(Serial& bus, PinName dir, int ID);

    /** Read info from the specified AX-12 register
    *
    * @param ID the ID of the AX-12 servo
    * @param start the address of the 1st register to be read
    * @param length the number of bytes to read
    * @param data where the read data is stored
    * @returns servo status
    */
    int read(int ID, int start, int length, char* data);

    /** Write info to the specified AX-12 register
    *
    * @param ID the ID of the AX-12 servo
    * @param start the address of the 1st register to be written to
    * @param length the number of bytes to write
    * @param data where the data to be written is stored
    * @returns servo status
    */
    int write(int ID, int start, int length, char* data);

    /** Sets the goal position for the servo
    *
    * @param degrees the new target position
    * @returns error code
    */
    int SetGoal(int degrees);

    /** Getss the current position for the servo
    *
    * @returns current position
    */
    float GetPosition();

private :
    Serial& _bus;
    DigitalOut _dir;
    int _ID;
};
#endif