#include "SpeedController.h"

using namespace std;

const float SpeedController::TS = 0.001f;                       // period of 1 ms
const float SpeedController::LOWPASS_FILTER_FREQUENCY = 100.0f; // given in [rad/s]
const float SpeedController::MIN_DUTY_CYCLE = 0.0f;             // minimum duty-cycle
const float SpeedController::MAX_DUTY_CYCLE = 1.0f;             // maximum duty-cycle

SpeedController::SpeedController(float counts_per_turn, float kn, float max_voltage, FastPWM& pwm, EncoderCounter& encoderCounter) : pwm(pwm), encoderCounter(encoderCounter), thread(osPriorityHigh, 4096)
{
    this->counts_per_turn = counts_per_turn;
    this->kn = kn;
    this->kp = 0.1f;
    this->max_voltage = max_voltage;

    // Initialisieren der PWM Ausgaenge
    pwm.period(0.00005); // PWM Periode von 50 us
    pwm.write(0.5);      // Duty-Cycle von 50%

    // Initialisieren von lokalen Variabeln
    previousValueCounter = encoderCounter.read();
    speedFilter.setPeriod(TS);
    speedFilter.setFrequency(LOWPASS_FILTER_FREQUENCY);
    desiredSpeed = 0.0f;
    actualSpeed = 0.0f;
    // actualAngle = 0.0f;

    // Starten des periodischen Tasks
    thread.start(callback(this, &SpeedController::run));
    ticker.attach(callback(this, &SpeedController::sendThreadFlag), TS);
}

SpeedController::SpeedController(float counts_per_turn, float kn, float kp, float max_voltage, FastPWM& pwm, EncoderCounter& encoderCounter) : pwm(pwm), encoderCounter(encoderCounter), thread(osPriorityHigh, 4096)
{
    this->counts_per_turn = counts_per_turn;
    this->kn = kn;
    this->kp = kp;
    this->max_voltage = max_voltage;

    // Initialisieren der PWM Ausgaenge
    pwm.period(0.00005); // PWM Periode von 50 us
    pwm.write(0.5);      // Duty-Cycle von 50%

    // Initialisieren von lokalen Variabeln
    previousValueCounter = encoderCounter.read();
    speedFilter.setPeriod(TS);
    speedFilter.setFrequency(LOWPASS_FILTER_FREQUENCY);
    desiredSpeed = 0.0f;
    actualSpeed = 0.0f;
    // actualAngle = 0.0f;

    // Starten des periodischen Tasks
    thread.start(callback(this, &SpeedController::run));
    ticker.attach(callback(this, &SpeedController::sendThreadFlag), TS);
}

SpeedController::~SpeedController()
{
    ticker.detach(); // Stoppt den periodischen Task
}

void SpeedController::setDesiredSpeedRPM(float desiredSpeed)
{
    this->desiredSpeed = desiredSpeed;
}

float SpeedController::getSpeedRPM()
{
    return actualSpeed;
}

void SpeedController::setDesiredSpeedRPS(float desiredSpeed)
{
    this->desiredSpeed = desiredSpeed*60.0f;
}

float SpeedController::getSpeedRPS()
{
    return actualSpeed/60.0f;
}

void SpeedController::run()
{
    while(true) {

        // wait for the periodic signal
        ThisThread::flags_wait_any(threadFlag);

        // calculate actual speed of motors in [rpm]
        short valueCounter = encoderCounter.read();
        short countsInPastPeriod = valueCounter - previousValueCounter;
        previousValueCounter = valueCounter;
        actualSpeed = speedFilter.filter((float)countsInPastPeriod/counts_per_turn/TS*60.0f);
        // actualAngle = actualAngle + actualSpeed/60.0f*TS;

        // calculate motor phase voltages
        float voltage = kp*(desiredSpeed - actualSpeed) + desiredSpeed/kn;

        // calculate, limit and set duty cycles
        float dutyCycle = 0.5f + 0.5f*voltage/max_voltage;
        if (dutyCycle < MIN_DUTY_CYCLE) dutyCycle = MIN_DUTY_CYCLE;
        else if (dutyCycle > MAX_DUTY_CYCLE) dutyCycle = MAX_DUTY_CYCLE;
        pwm.write(static_cast<double>(dutyCycle));

    }
}

void SpeedController::sendThreadFlag()
{
    thread.flags_set(threadFlag);
}