#include "mbed.h"
#include <string>
#include "C12832.h"
#include "ENGO333_MMA7660.h"
#include "ENGO333_MPU9150.h"
#include "Tiltmeter.h"


enum AccelType
{
    MMA7660,    // MMA7660 accelerometer on the mbed application board
    MPU9150     // MPU9150 IMU (external to mbed application board)
};


// Function to prompt the user to select between the MMA7660 accelerometer or
// the MPU9150 IMU. The user can select between these choices by moving the 
// joystick left or right; they make their selection by pressing the joystick
// button.
//
// Arguments:
//    lcd = The LCD screen to use for display
//    joystick = Object linked to the joystick to be used
//
// Returns:
//    The function returns an enumeration identifying the sensor selected
AccelType SelectAccelerometer( C12832& lcd, BusIn& joystick );


int main()
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // START: Setup/variable declaration


    // Create LCD object for LCD screen operations
    C12832 lcd(p5, p7, p6, p8, p11);

    // Create digital outputs for LED1 and LED2 that will be used to indicate
    // if things are working properly -- both values are to LOW (off)
    DigitalOut led1(LED1);
    DigitalOut led2(LED2);
    led1 = 0;
    led2 = 0;

    // Create bus input for joystick
    BusIn joystick(p13, p16, p14);     // Binary [ Centre | Right | Left ] joystick
    
    
    // END: Setup/variable declaration
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // START: Take Measurements
    
    
    // Create an accelerometer object
    ENGO333_MMA7660 accelerometer;  
    
    
    // Detect sensor
    if( accelerometer.TestConnection() ) 
    {
        // Turn on the first LED to show that the connection was detected
        led1 = 1;
    } 
    else 
    {
        lcd.locate(1, 11);
        lcd.printf("ERROR: Unknown sensor");
        return 1;
    }


    // Prepare the screen
    lcd.cls();
    lcd.locate(1, 1);
    lcd.printf("Press joystick to measure");

    
    // Variable to store the value/status of the joystick. It is initialized
    // to represent pushing the joystick button. Since the loop below 
    while( true ) 
    {
        // Check if the joystick button is being pressed
        if (joystick.read() == 4) 
        {
            // Turn on second LED to indicate a measurement is being made
            led2 = 1;
            
            
            // Get the tilt angles
            accelerometer.ComputeTiltAngles();            
            
            
            // Print the roll
            lcd.locate(1, 11);
            lcd.printf("Roll (deg)");
            lcd.locate(50, 11);
            lcd.printf(": % 8.2f", accelerometer.GetRoll());
            
            // Print the pitch
            lcd.locate(1, 21);
            lcd.printf("Pitch (deg)");
            lcd.locate(50, 21);
            lcd.printf(": % 8.2f", accelerometer.GetPitch());
            
            
            // Short delay to avoid multiple measurements being displayed
            wait_ms(250);


            // Turn off second LED to indicate a measurement is complete
            led2 = 0;
        }
        
    }
    
    
    // END: Take Measurements
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
}


AccelType SelectAccelerometer( C12832& lcd, BusIn& joystick )
{
    // This is the current choice
    AccelType choice = MMA7660;
    
    // This is the string to display the current choice (using asterisk)
    string sensorChoice = "MMA7660 *       MPU9150";
    
    // Print the default choice
    lcd.cls();
    lcd.locate(1, 1);
    lcd.printf("Move L/R to select sensor:");
    lcd.locate(1, 18);
    lcd.printf("%s", sensorChoice.c_str());

  
    // Create while-loop for user to select sensor
    while(1) 
    {
        // read the joystick
        int joy = joystick.read();
        
        
        if (joy == 2)       // joystick pushed to the right
        {                  
            sensorChoice = "MMA7660       * MPU9150";
            choice = MPU9150;
            lcd.locate(1, 18);
            lcd.printf("%s", sensorChoice.c_str());
        } 
        else if (joy == 1)  // joystick pushed to the left
        {
            sensorChoice = "MMA7660 *       MPU9150";
            choice = MMA7660;
            lcd.locate(1, 18);
            lcd.printf("%s", sensorChoice.c_str());
        } 
        else if (joy == 4)  // joystick "button" pushed (i.e., "select")
        {      
            break;
        } 
    }
    
    // return the current choice
    return choice;
}