/**
 * File : ENGO333_I2C.h
 * Created by : Chandra Tjhai
 * Created on : September 10, 2016
 *
 * Description :
 * This file contains a class to handle I2C bus operation for ENGO 333.
 */

#ifndef ENGO333_I2C_H
#define ENGO333_I2C_H

#include "mbed.h"

/**
 * Class
 *  A class to handle I2C read/write operation
 */
class ENGO333_I2C
{
private:
    I2C i2c;    // Object for handling I2C bus operations
    
public:
    /**
     * Constructor
     *
     * Input
     *  sda = SDA or I2C serial data pin
     *  scl = SCL or I2C serial clock pin
     */
    ENGO333_I2C(PinName sda, PinName scl);
    
    /**
     * Destructor
     */
    ~ENGO333_I2C();
    
    /**
     * Function :
     *  Set I2C bus frequency
     *
     * Input :
     *  freq = desired frequency in Hz
     */
    void setSpeed(int freq);
    
    /**
     * Function :
     *  Write one byte data into device's register
     *
     * Input :
     *  deviceAddress = Device's I2C address
     *  registerAddress = Device's register address to be written on
     *  value = Value to be written on device's register address
     */
    void writeOneByte(char deviceAddress, char registerAddress, char value);
    
    /**
     * Function :
     *  Read one byte from device's register address
     *
     * Input :
     *  deviceAddress = Device's I2C address
     *  registerAddress = Device's register address to be read
     *
     * Return :
     *  One byte data read from device's register
     */
    char readOneByte(char deviceAddress, char registerAddress);
    
    /**
     * Function :
     *  Read multiple bytes from device's register address
     *
     * Input :
     *  deviceAddress = Device's I2C address
     *  registerAddress = Device's register address to be read
     *  value = Array of data to store values read from device's register
     */
    void readBytes(char deviceAddress, char registerAddress, char* value, int length);
};

#endif
