/**
 * File : ENGO333_MPU9150.h
 * Created by : Chandra Tjhai
 * Created on : September 19, 2016
 *
 * Description :
 * This library is created for ENGO 333 class. The inertial sensor needs to be 
 * mounted on the mbed Application Board. The I2C connection is automatically 
 * set to p28 (SDA) and p27 (SCL).
 */
 
#ifndef ENGO333_MPU9150_H
#define ENGO333_MPU9150_H

#include "ENGO333_I2C.h"
#include "Tiltmeter.h"

/**
 * Define Macros: Device I2C slave addresses (R/W)
 */
#define MPU9150_ADDRESS         (0x68 << 1)

 /**
 * Define Macros: Configuration Registers
 */
#define MPU9150_CONFIG_REG              0x1A
#define MPU9150_ACCEL_CONFIG_REG        0x1C
#define MPU9150_INT_STATUS_REG          0x3A
#define MPU9150_INT_PIN_CFG_REG         0x37
#define MPU9150_PWR_MGMT_1_REG          0x6B
#define MPU9150_WHO_AM_I_REG            0x75

/**
 * Define Macros: Measurement Data Registers
 */
#define MPU9150_ACCEL_XOUT_H_REG        0x3B
#define MPU9150_ACCEL_YOUT_H_REG        0x3D
#define MPU9150_ACCEL_ZOUT_H_REG        0x3F

/**
 * Define Macros: IMU Definitions
 */
#define MPU9150_SLEEP_BIT               6
#define MPU9150_ACCEL_RANGE_2G          0
#define MPU9150_ACCEL_RANGE_4G          1
#define MPU9150_ACCEL_RANGE_8G          2
#define MPU9150_ACCEL_RANGE_16G         3
#define MPU9150_I_AM                    0x68
#define MPU9150_I2C_FAST_MODE           400000
#define MPU9150_I2C_STD_MODE            100000

/**
 * Class
 *  A class to handle MPU-9150 9-DOF sensor
 */
class ENGO333_MPU9150
{
private:
    ENGO333_I2C i2c;
    float measAccel[3];  // Measured acceleration values in units of m/s/s

public:
    /**
     * Default Constructor
     *  Once called, triggering device initialization and set data variables to 
     *  zero. Accelerometer is set to +-2G by default.
     */
    ENGO333_MPU9150();
    
    /**
     * Function :
     *  Enable/disable device sleep mode
     * 
     * Argument :
     *  state = TRUE/FALSE
     *
     * Return :
     *  NONE
     */
    void setSleepMode(bool state);
    
    /**
     * Function :
     *  Test device's accelerometer connection
     * 
     * Argument :
     *  NONE
     *
     * Return :
     *  Return TRUE if connection is good, otherwise FALSSE
     */
    virtual bool TestConnection();
    
     /**
     * Function :
     *  Set accelerometer full scale range, see MPU9150_ACCEL_RANGE_XG
     * 
     * Argument :
     *  range = values of MPU9150_ACCEL_RANGE_XG
     *
     * Return :
     *  NONE
     */
    void setAccelRange(char range);
    
    /**
     * Function :
     *  Read raw accelerometer data, 3 axes
     * 
     * Argument :
     *  NONE
     *
     * Return :
     *  NONE
     */
    virtual void ReadAccelerometers();
    
     /**
     * Function :
     *  Get raw X-axis acceleration
     * 
     * Argument :
     *  NONE
     *
     * Return :
     *  Raw X-axis acceleration
     */
    virtual float GetAccelX() const;
    
    /**
     * Function :
     *  Get raw Y-axis acceleration
     * 
     * Argument :
     *  NONE
     *
     * Return :
     *  Raw Y-axis acceleration
     */
    virtual float GetAccelY() const;
    
    /**
     * Function :
     *  Get raw Z-axis acceleration
     * 
     * Argument :
     *  NONE
     *
     * Return :
     *  Raw Z-axis acceleration
     */
    virtual float GetAccelZ() const;
    
};

#endif
 

