/**
 * File : ENGO333_MMA7660.h
 * Created by : Chandra Tjhai
 * Created on : September 10, 2016
 *
 * Description :
 * This library is created for ENGO 333 class. The accelerometer is mounted
 * on the mbed Application Board. The I2C connection is permanently set to p28
 * (SDA) and p27 (SCL).
 */

#ifndef ENGO333_MMA7660_H
#define ENGO333_MMA7660_H

#include "ENGO333_I2C.h"
#include "Tiltmeter.h"

// Sensor I2C address
#define MMA7660_ADDRESS 0x98

// Define sensor registers
#define MMA7660_XOUT_REG 0x00
#define MMA7660_YOUT_REG 0x01
#define MMA7660_ZOUT_REG 0x02
#define MMA7660_TILT_REG 0x03
#define MMA7660_SRST_REG 0x04
#define MMA7660_SPCNT_REG 0x05
#define MMA7660_INTSU_REG 0x06
#define MMA7660_MODE_REG 0x07
#define MMA7660_SR_REG 0x08
#define MMA7660_PDET_REF 0x09
#define MMA7660_PD_REG 0x0A

// Define acceleration scale
#define MMA7660_SCALE (21.33)

// Define AMSR values
typedef enum
{
    MMA7660_AMSR120 = 0,
    MMA7660_AMSR64 = 1,
    MMA7660_AMSR32 = 2,
    MMA7660_AMSR16 = 3,
    MMA7660_AMSR8 = 4,
    MMA7660_AMSR4 = 5,
    MMA7660_AMSR2 = 6,
    MMA7660_AMSR1 = 7
}MMA7660_AMSR_t;

/**
 * Class
 *  A class to handle MMA7660 3-DOF accelerometer
 */
class ENGO333_MMA7660
{
private:
    ENGO333_I2C i2c;    // I2C communication connection
    float measAccel[3];  // Measured acceleration values in units of m/s/s

public:
    /**
     * Default Constructor
     *  Once called, trigger active mode and set MMA7660_AMSR8
     */
    ENGO333_MMA7660();
    
    /**
     * Function :
     *  Test device's accelerometer connection. MMA7660 does not have identifier
     *  registers. Thus, this function will simply take measurements and check 
     *  their validity.
     *
     * Argument :
     *  NONE
     *
     * Return :
     *  Return TRUE if connection is good, otherwise FALSSE
     */
    virtual bool TestConnection();


private:

   // Read the accelerometer data and store the values for later use. You can access the values by
   // calling the GetAccelX(), GetAccelY() and/or GetAccelZ();
   //
   // Arguments:
   //    None
   //
   // Returns:
   //    Nothing
   //
   // Remarks:
   //    The data will be stored in the 'MeasuredAccel' member variable in units of m/s/s
   virtual void ReadAccelerometers();


   // Get the most recently measured X-axis acceleration as stored during the last call to
   // ReadAccelerometer()
   //
   // Arguments:
   //    None
   //
   // Returns:
   //    The function returns the most recently measured X-axis acceleration in units of m/s/s
   virtual float GetAccelX() const;


   // Get the most recently measured Y-axis acceleration as stored during the last call to
   // ReadAccelerometer()
   //
   // Arguments:
   //    None
   //
   // Returns:
   //    The function returns the most recently measured Y-axis acceleration in units of m/s/s
   virtual float GetAccelY() const;


   // Get the most recently measured Z-axis acceleration as stored during the last call to
   // ReadAccelerometer()
   //
   // Arguments:
   //    None
   //
   // Returns:
   //    The function returns the most recently measured Z-axis acceleration in units of m/s/s
   virtual float GetAccelZ() const;
   
   
   // Set the device to 'active' mode
   //
   // Arguments: 
   //    None
   //
   // Returns:
   //    Nothing
   void SetActiveMode();
   
   
   // Set the device sampling rate through AM bits, see MMA7660_SR_REG
   //
   // Arguments:
   //    samplingRate = setting of AM bits in MMA7660_SR_REG
   //
   // Returns:
   //    Nothing
   void SetSamplingRateAM(MMA7660_AMSR_t samplingRate);
    
};

#endif

