/*
  Stnseg.cpp - mbed library for 2/4/8 digit Sixteen (16) segment LED driver.
  Copyright 2015 by morecat_lab
   
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*/

#ifndef STNSEG_H
#define STNSEG_H

#include "mbed.h"
#include <Timer.h>

// the order of segment bit is as follows (16 bit)
// |             |             |               |            |
//  A1, A2, B, C, D1, D2, E, F,  G1, G2, H, I,  J, K, L, M
// note that DP must set independent port

#define NUM_PAT16_0 0xff09
#define NUM_PAT16_1 0x3000
#define NUM_PAT16_2 0xeec0
#define NUM_PAT16_3 0xfcc0
#define NUM_PAT16_4 0x31c0
#define NUM_PAT16_5 0xddc0
#define NUM_PAT16_6 0xdfc0
#define NUM_PAT16_7 0xf000
#define NUM_PAT16_8 0xffc0
#define NUM_PAT16_9 0xfdc0
#define NUM_PAT16_A 0xf3c0
#define NUM_PAT16_B 0xfc52
#define NUM_PAT16_C 0xcf00
#define NUM_PAT16_D 0xfc12
#define NUM_PAT16_E 0xcf80
#define NUM_PAT16_F 0xf310

#define NUM_PAT16_MINUS 0x00c0

/**
 * Sixteen segment LED driver library
 */
class Stnseg
{
private:
    unsigned long _lastUpdateTime;
    int _updateInterval;
    DigitalOut _dataPin, _clockPin, _latchPin, _dpPin;
    BusOut _digPins;
    uint32_t _buffer[8];
    int _numOfDigs;
    int _dig;       // support 4, 6 or 8
    bool _zeroSupress;
    bool _kcommon;  // Cathode-common flag
    Timer timer;
public:
    static const uint16_t numConv[16];

    /**
     * create an 4 digit sixteen segment driver
     *
     * @param PinName data (for 74HC959)
     * @param PinName clock (for 74HC959)
     * @param PinName latch (for 74HC959)
     * @param PinName dp Pin No for segment DP
     * @param PinName d1 Pin No for dight 1
     * @param PinName d2 Pin No for dight 2
     * @param PinName d3 Pin No for dight 3
     * @param PinName d4 Pin No for dight 4
     */
    Stnseg(PinName data, PinName clock, PinName latch, PinName dp,
         PinName d1,PinName d2, PinName d3, PinName d4);

    /**
     * create an 6 digit sixteen segment driver
     *
     * @param PinName data (for 74HC959)
     * @param PinName clock (for 74HC959)
     * @param PinName latch (for 74HC959)
     * @param PinName dp Pin No for segment DP
     * @param PinName d1 Pin No for dight 1
     * @param PinName d2 Pin No for dight 2
     * @param PinName d3 Pin No for dight 3
     * @param PinName d4 Pin No for dight 4
     * @param PinName d4 Pin No for dight 5
     * @param PinName d4 Pin No for dight 6
     */
    Stnseg(PinName data, PinName clock, PinName latch, PinName dp,
         PinName d1,PinName d2, PinName d3, PinName d4,
         PinName d5, PinName d6);

    /**
     * create an 8 digit sixteen segment driver
     *
     * @param PinName data (for 74HC959)
     * @param PinName clock (for 74HC959)
     * @param PinName latch (for 74HC959)
     * @param PinName dp Pin No for segment DP
     * @param PinName d1 Pin No for dight 1
     * @param PinName d2 Pin No for dight 2
     * @param PinName d3 Pin No for dight 3
     * @param PinName d4 Pin No for dight 4
     * @param PinName d5 Pin No for dight 5
     * @param PinName d6 Pin No for dight 6
     * @param PinName d7 Pin No for dight 7
     * @param PinName d8 Pin No for dight 8
     */
    Stnseg(PinName data, PinName clock, PinName latch, PinName dp,
         PinName d1,PinName d2, PinName d3, PinName d4,
         PinName d5,PinName d6, PinName d7, PinName d8);
    
    /**
     * start driver
     */
    void begin(void);

    /**
     * use Kathode Common LED
     */
    void setKcommon(void);
    
    /**
     * use Anode Common LED (default)
     */
    void setAcommon(void);
    
    /**
     * get a charcter pattern from a number
     *
     * @param i number
     *
     * @returns bit pattern of number i
     *
     */
    uint16_t segCh(char i);

    /**
     * turn on DP 
     *
     * @param d dight
     *
     */
    void setDot(int d);

    /**
     * turn off DP 
     *
     * @param d dight
     *
     */
    void clearDot(int d);
    
    /**
     * write a number to LED 
     *
     * @param d number
     *
     */
    void writeNum(int n);

    /**
     * write a number to 4 dight LED 
     *
     * @param n number
     *
     */
    void writeNum4(int n);

    /**
     * write a number to 6 dight LED 
     *
     * @param n number
     *
     */
    void writeNum6(int n);

    /**
     * write a number to 8 dight LED 
     *
     * @param n number
     *
     */
    void writeNum8(int n);
    
    /**
     * write numbers to each dight of 2 dight LED 
     *
     * @param d1 digit 1 number
     * @param d2 digit 2 number
     *
     */
    void writeNum(char d1, char d2);

    /**
     * write numbers to each dight of 4 dight LED 
     *
     * @param d1 digit 1 number
     * @param d2 digit 2 number
     * @param d3 digit 3 number
     * @param d4 digit 4 number
     *
     */
    void writeNum(char d1, char d2, char d3, char d4);

    /**
     * write numbers to each dight of 8 dight LED 
     *
     * @param d1 digit 1 number
     * @param d2 digit 2 number
     * @param d3 digit 3 number
     * @param d4 digit 4 number
     * @param d5 digit 5 number
     * @param d6 digit 6 number
     * @param d7 digit 7 number
     * @param d8 digit 8 number
     *
     */
    void writeNum(char d1, char d2, char d3, char d4, char d5, char d6, char d7, char d8);

    /**
     * zero supress: tell driver not to display 0 in the left
     *
     */
    void supressZero();

    /**
     * control zero supress bit
     *
     * @param t, 1:supress on, 0:supress off
     *
     */
    void setZeroSupress(bool t);

    /**
     * write hex number to LED
     *
     * @param n (long)number
     *
     */
    void writeHex(long n);

    
    /**
     * write patterns to each dight of 4 dight LED 
     *
     * @param d1 digit 1 pattern
     * @param d2 digit 2 pattern
     * @param d3 digit 3 pattern
     * @param d4 digit 4 pattern
     *
     */
    void writeRawData(uint32_t d1, uint32_t d2, uint32_t d3, uint32_t d4);

    /**
     * write patterns to each dight of 6 dight LED 
     *
     * @param d1 digit 1 pattern
     * @param d2 digit 2 pattern
     * @param d3 digit 3 pattern
     * @param d4 digit 4 pattern
     * @param d5 digit 5 pattern
     * @param d6 digit 6 pattern
     *
     */
    void writeRawData(uint32_t d1, uint32_t d2, uint32_t d3, uint32_t d4, uint32_t d5, uint32_t d6);

    /**
     * write patterns to each dight of 8 dight LED 
     *
     * @param d1 digit 1 pattern
     * @param d2 digit 2 pattern
     * @param d3 digit 3 pattern
     * @param d4 digit 4 pattern
     * @param d5 digit 5 pattern
     * @param d6 digit 6 pattern
     * @param d7 digit 7 pattern
     * @param d8 digit 8 pattern
     *
     */
    void writeRawData(uint32_t d1, uint32_t d2, uint32_t d3, uint32_t d4, uint32_t d5, uint32_t d6, uint32_t d7, uint32_t d8);

    /**
     * write patterns to a dight 
     *
     * @param d digit
     *
     * @param value pattern
     *
     */
    void write(uint8_t d, uint32_t value);

    /**
     * Clear LED buffer
     */
    void clear(void);
    
    /**
     * Turn off LED
     */
    void turnOff(void);

    /**
     * Turn on LED
     */
    void turnOn(void);

    /**
     * Update One dight of LED
     */
    void updateSeg(void);

    /**
     * keep updating LED for specified period
     *
     * @param ms period (ms)
     *
     */
    void updateWithDelay(int ms);

    /**
     * Update LED Once with 1ms delay
     */
    void updateOnce(void);
};

#endif  // Stnseg.h
