/*
  SpiLED.h - mbed library for Serial seven segment LED driver.
  This library is for Serial-in pararel-out shift register (HC595) and 7 Segiemnt LED module 
  Copyright 20145 by morecat_lab
   
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*/

#ifndef SPI7SEG_H
#define SPI7SEG_H

#include <Timer.h>
#include "mbed.h"

// LED PIN ASSIGN
//
// P G F E D C B A
//
//   A
//   --
// F|__|B
// E|G |C
//   --   .P
//   D

#define NUM_PAT_0 0xc0
#define NUM_PAT_1 0xf9
#define NUM_PAT_2 0xa4
#define NUM_PAT_3 0xb0
#define NUM_PAT_4 0x99
#define NUM_PAT_5 0x92
#define NUM_PAT_6 0x82
#define NUM_PAT_7 0xf8
#define NUM_PAT_8 0x80
#define NUM_PAT_9 0x90
#define NUM_PAT_A 0x88
#define NUM_PAT_B 0x83
#define NUM_PAT_C 0xc6
#define NUM_PAT_D 0xa1
#define NUM_PAT_E 0x86
#define NUM_PAT_F 0x8e

/**
 * Spi Seven segment LED driver library (8 digit)
 */
class Spi7Seg
{
private:
    DigitalOut _dataPin, _clockPin, _latchPin;
    char _buffer[8];
    int _numOfDigs;
    bool _zeroSupress;
    Timer timer;
public:
    static const int numConv[16];
    /**
     * create an 8 digit SPI driven seven segment driver
     *
     * @param PinName a Pin No for segment data
     * @param PinName b Pin No for segment clock
     * @param PinName c Pin No for segment latch
     */
    Spi7Seg(PinName data,PinName clock,PinName latch);
   
    /**
     * start driver
     */
    void begin(void);

    /**
     * get a charcter pattern from a number
     *
     * @param i number
     *
     * @returns bit pattern of number i
     *
     */
    char segCh(char i);

    /**
     * turn on DP 
     *
     * @param d dight
     *
     */
    void setDot(int d);

    /**
     * turn off DP 
     *
     * @param d dight
     *
     */
    void clearDot(int d);
    
    /**
     * write a number to LED 
     *
     * @param n number
     *
     */
    void writeNum(long n);

    /**
     * write numbers to each dight of 8 dight LED 
     *
     * @param d1 digit 1 number
     * @param d2 digit 2 number
     * @param d3 digit 3 number
     * @param d4 digit 4 number
     * @param d5 digit 5 number
     * @param d6 digit 6 number
     * @param d7 digit 7 number
     * @param d8 digit 8 number
     *
     */
    void writeNum(char d1, char d2, char d3, char d4, char d5, char d6, char d7, char d8);

    /**
     * zero supress: tell driver not to display 0 in the left
     *
     */
    void supressZero();

    /**
     * control zero supress bit
     *
     * @param t, 1:supress on, 0:supress off
     *
     */
    void setZeroSupress(bool t);

    /**
     * write hex number to LED
     *
     * @param n (long)number
     *
     */
    void writeHex(long n);

    /**
     * write patterns to each dight of 8 dight LED 
     *
     * @param d1 digit 1 pattern
     * @param d2 digit 2 pattern
     * @param d3 digit 3 pattern
     * @param d4 digit 4 pattern
     * @param d5 digit 5 pattern
     * @param d6 digit 6 pattern
     * @param d7 digit 7 pattern
     * @param d8 digit 8 pattern
     *
     */
    void writeRawData(char d1, char d2, char d3, char d4, char d5, char d6, char d7, char d8);

    void write(uint8_t x, uint8_t y, uint8_t value);

    /**
     * write patterns to a dight 
     *
     * @param d digit
     *
     * @param value pattern
     *
     */
    void write(uint8_t d, uint8_t value);
 
    /**
     * Clear LED buffer
     */
    void clear(void);
    
    /**
     * Update LED (by internal clock)
     *
     * @returns sync = 1, if digit == 0
     *
     */
    void update(void);

    /**
     * keep updating LED for specified period
     *
     * @param ms period (ms)
     *
     */
    void updateWithDelay(int ms);

};

#endif  // SPI7SEG.h
