/*
  Spi7Seg.h - mbed library for Serial seven segment LED driver.
  This library is for Serial-in pararel-out shift register (HC595) and 7 Segiemnt LED module 
  Copyright 20154 by morecat_lab
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
*/

#include <Timer.h>
#include "Spi7Seg.h"

const int Spi7Seg::numConv[] = {
    NUM_PAT_0, NUM_PAT_1, NUM_PAT_2, NUM_PAT_3,
    NUM_PAT_4, NUM_PAT_5, NUM_PAT_6, NUM_PAT_7,
    NUM_PAT_8, NUM_PAT_9, NUM_PAT_A, NUM_PAT_B,
    NUM_PAT_C, NUM_PAT_D, NUM_PAT_E, NUM_PAT_F};

// 8 digit
Spi7Seg::Spi7Seg(PinName data,PinName clock, PinName latch) :
    _dataPin(data), _clockPin(clock), _latchPin(latch) {
    _numOfDigs = 8;
    _dataPin = 0;
    _clockPin = 1;
    _latchPin = 1;
    _zeroSupress = 1;
}

void Spi7Seg::begin(void) {
    clear();
}

char Spi7Seg::segCh(char i) {
    return Spi7Seg::numConv[i];
}

void Spi7Seg::setDot(int d) {
    _buffer[d] &= 0x7f;   // P <- 0
}

void Spi7Seg::clearDot(int d) {
    _buffer[d] |= 0x80;  // P <- 1
}

void Spi7Seg::writeNum(long n) {
    _buffer[0] = segCh((n % 100000000) / 10000000);
    _buffer[1] = segCh((n % 10000000) / 1000000);
    _buffer[2] = segCh((n % 1000000) / 100000);
    _buffer[3] = segCh((n % 100000) / 10000);
    _buffer[4] = segCh((n % 10000) / 1000);
    _buffer[5] = segCh((n % 1000) / 100);
    _buffer[6] = segCh((n % 100) / 10);
    _buffer[7] = segCh(n % 10);
    Spi7Seg::supressZero();
}

void Spi7Seg::writeNum(char d1, char d2, char d3, char d4,
                    char d5, char d6, char d7, char d8)
{
    _buffer[0] = segCh(d1);
    _buffer[1] = segCh(d2);
    _buffer[2] = segCh(d3);
    _buffer[3] = segCh(d4);
    _buffer[4] = segCh(d5);
    _buffer[5] = segCh(d6);
    _buffer[6] = segCh(d7);
    _buffer[7] = segCh(d8);
    Spi7Seg::supressZero();
}

void Spi7Seg::writeHex(long n) {
    _buffer[0] = segCh((n >> 28) & 0xf);
    _buffer[1] = segCh((n >> 24) & 0xf);
    _buffer[2] = segCh((n >> 20) & 0xf);
    _buffer[3] = segCh((n >> 16) & 0xf);
    _buffer[4] = segCh((n >> 12) & 0xf);
    _buffer[5] = segCh((n >> 8) & 0xf);
    _buffer[6] = segCh((n >> 4) & 0xf);
    _buffer[7] = segCh(n & 0xf);
    Spi7Seg::supressZero();
}

void Spi7Seg::setZeroSupress(bool t) {
    _zeroSupress = t;
}

void Spi7Seg::supressZero() {
    if (_zeroSupress ) {
        for (int i = 0 ; i < (_numOfDigs-1) ; i++) {
            if (_buffer[i] == segCh(0)) {
                _buffer[i] |= 0x7f; // blank except dot
            } else {
                break;
            }
        }
    }
}


void Spi7Seg::writeRawData(char d1, char d2, char d3, char d4,
                        char d5, char d6, char d7, char d8) {
    _buffer[0] = d1;
    _buffer[1] = d2;
    _buffer[2] = d3;
    _buffer[3] = d4;
    _buffer[4] = d5;
    _buffer[5] = d6;
    _buffer[6] = d7;
    _buffer[7] = d8;
}

void Spi7Seg::write(uint8_t d, uint8_t value) {
    _buffer[d] = value;
}

void Spi7Seg::clear(void) {
    for (int i = 0; i < 8; i++){
        _buffer[i] = 0;
    }
}


void Spi7Seg::update(void) {
    _latchPin = 0;
    for (int col = _numOfDigs ; col >= 0 ; col--) {  // reverse order
        for (int i = 7; i >= 0; i--) {
            _clockPin = 0;
            if(_buffer[col] & (1 << i)){
                _dataPin = 1;
            } else {
                _dataPin = 0;
            }
            _clockPin = 1;
            _dataPin = 0;
        }
    }
    _latchPin = 1;
}

void Spi7Seg::updateWithDelay(int ms) {
    timer.reset();  // to avoid overflow 32bit counter  (~=30min)
    int start = timer.read_ms();
    update();
    do {
        ;
    } while((timer.read_ms() - start) < ms);
}

// EOF
