/**
 * File:      dins-r8.cpp
 * 
 * Author:    Modtronix Engineering - www.modtronix.com
 * 
 * Description:
 * 
 * Software License Agreement:
 * This software has been written or modified by Modtronix Engineering. The code
 * may be modified and can be used free of charge for commercial and non commercial
 * applications. If this is modified software, any license conditions from original
 * software also apply. Any redistribution must include reference to 'Modtronix
 * Engineering' and web link(www.modtronix.com) in the file header. 
 * 
 * THIS SOFTWARE IS PROVIDED IN AN 'AS IS' CONDITION. NO WARRANTIES, WHETHER EXPRESS,
 * IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE
 * COMPANY SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 */ 

#include "mbed.h"
#include "dins-r8_i2c.h"
#include <algorithm>    // std::min


// DEFINES ////////////////////////////////////////////////////////////////////
/** Write to all relays. Each bit of byte is 1 relayCursor moves backwards, no parameters */
#define CMD_WRITE_ALL_RELAYS            0x80


// GLOBAL VARIABLES ///////////////////////////////////////////////////////////


// Function Prototypes ////////////////////////////////////////////////////////


DINS_R8_I2C::DINS_R8_I2C(I2C* i2cBus) {
    pI2C = i2cBus;
    i2cAdr = 0x60;      //Default I2C address of the LCD
    relays = 0;         //All relays off
}


/** Writes given relay with given value
 *
 * @param relayNumber Relay to update, number from 1 to 8
 *
 * @param value New value, 0 if off, 1 is on
 *
 * @returns 0 if success, else error code
 */
uint8_t DINS_R8_I2C::writeRelay(int8_t relayNumber, bool value) {
    uint8_t mask;
    uint8_t relaysNew;

    relayNumber--;
    if(relayNumber>7) {
        return DINS_R8_ERR_INVALID_RELAY;       //Error, invalid Relay Number
    }
    mask = 0x01 << relayNumber;

    //Clear given relay
    relaysNew = relays & (~mask);
    if(value==true) {
        relaysNew |= mask;  //Set given relay
    }

    return writeAllRelays(relaysNew);
}


/** Set all relays with new values. Each bit is 1 relay
 *
 * @param relays New values for relays, each bit is 1 relay
 *
 * @returns 0 if success, else error code
 */
uint8_t DINS_R8_I2C::writeAllRelays(int8_t relays) {
    char cmd[4];

    if(this->relays != relays) {

        this->relays = relays;

        //- First byte of data is 0x80 ("Write All Relays" command for DINS-8R board)
        //- Second byte of data is 0x0f (value of relays for DINS-8R board)
        cmd[0] = CMD_WRITE_ALL_RELAYS;
        cmd[1] = relays;
        return pI2C->write(i2cAdr, cmd, 2);

    }

    return 0;   //Success
}



