#include "mbed.h"
#include "motordriver.h"    //Library to drive motors
#include "PID.h"            //PID library for distance control
#include "LSM9DS1.h"
#define PI 3.14159
// Earth's magnetic field varies by location. Add or subtract
// a declination to get a more accurate heading. Calculate
// your's here:
// http://www.ngdc.noaa.gov/geomag-web/#declination
#define DECLINATION -4.94 // Declination (degrees) in Atlanta,GA.

Motor LeftM(p21, p22, p23,1); // pwm, fwd, rev
Motor RightM(p26, p27, p24,1);

InterruptIn rhes(p15);
InterruptIn lhes(p16);

PID leftPid(1.0, 0.0, 0.0,0.01);  //Kc, Ti, Td
PID rightPid(1.0, 0.0, 0.0,0.01); //Kc, Ti, Td

Serial pc(USBTX, USBRX);

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);

LSM9DS1 IMU(p9, p10, 0xD6, 0x3C);

PwmOut speaker(p25);

int countl = 0, countr = 0;
int xx=0,yy=0;

float note1 = 1568.0, note2 = 1396.9, note3 = 1244.5;
// Calculate pitch, roll, and heading.
// Pitch/roll calculations taken from this app note:
// http://cache.freescale.com/files/sensors/doc/app_note/AN3461.pdf?fpsp=1
// Heading calculations taken from this app note:
// http://www51.honeywell.com/aero/common/documents/myaerospacecatalog-documents/Defense_Brochures-documents/Magnetic__Literature_Application_notes-documents/AN203_Compass_Heading_Using_Magnetometers.pdf
float getHeading(float mx, float my, float mz)
{
// touchy trig stuff to use arctan to get compass heading (scale is 0..360)
    mx = -mx;
    float heading;
    if (my == 0.0)
        heading = (mx < 0.0) ? 180.0 : 0.0;
    else
        heading = atan2(mx, my)*360.0/(2.0*PI);
    heading -= DECLINATION; //correct for geo location
    if(heading>180.0) heading = heading - 360.0;
    else if(heading<-180.0) heading = 360.0 + heading;
    else if(heading<0.0) heading = 360.0  + heading;
    heading = fabs(heading);
    
    return heading;
}

float getHead(){
    while(!IMU.magAvailable(X_AXIS));
        IMU.readMag();
    return getHeading(IMU.calcMag(IMU.mx), IMU.calcMag(IMU.my), IMU.calcMag(IMU.mz));
}

void leftM_count() {
    countl++;
}
void rightM_count() {
    countr++;
}

void move(float dist, int dir = 1){
    led1 = 1;
    
    speaker.period(1.0/note1);
    speaker =0.25;
    wait(0.3);
    speaker.period(1.0/note2);
    wait(0.3);
    speaker=0.0;
    
    leftPid.setInputLimits(0, 1000);
    leftPid.setOutputLimits(0.0, 0.9);
    leftPid.setMode(AUTO_MODE);
    rightPid.setInputLimits(0, 1000);
    rightPid.setOutputLimits(0.0, 0.9);
    rightPid.setMode(AUTO_MODE);
    
    int leftPulses      = 0;
    int leftPrevPulses  = 0;
    float leftVelocity  = 0.0;
    int rightPulses     = 0;
    int rightPrevPulses = 0;
    float rightVelocity = 0.0;
    
    wait(1);
    
    leftPid.setSetPoint(750);
    rightPid.setSetPoint(750);
    
    while ((leftPulses < dist) || (rightPulses < dist)) {
        leftPulses = countl;
        leftVelocity = (leftPulses - leftPrevPulses) / 0.01;
        leftPrevPulses = leftPulses;
        leftPid.setProcessValue(leftVelocity);
        LeftM.speed(leftPid.compute()*dir);
        
        rightPulses = countr;
        rightVelocity = (rightPulses - rightPrevPulses) / 0.01;
        rightPrevPulses = rightPulses;
        rightPid.setProcessValue(rightVelocity);
        RightM.speed(rightPid.compute()*dir);
        
        wait(0.01);
    }
    countl = 0;
    countr = 0;
    RightM.speed(0);
    LeftM.speed(0);
    led1 = 0;
    wait(1);
}

void turn(int direction){
    led2 = 1;
    
    speaker.period(1.0/note2);
    speaker =0.25;
    wait(0.3);
    speaker.period(1.0/note1);
    wait(0.3);
    speaker=0.0;
    
    float head = 0, oldHead = 0,newHead = 0;
    oldHead = getHead();
    head = oldHead;
    //printf("Old Heading: %f\n\r", oldHead);
    if (direction == 1){
        newHead = oldHead + 90;
        if (newHead > 360)
            newHead -= 360;
    }
    else{
        newHead = oldHead - 90;
        if (newHead < 0)
            newHead += 360;
    }
    //printf("New Heading: %f\n\r", newHead);
    
    while ((head < newHead - 1) || (head > newHead + 1)){
        if ((head < newHead && (newHead - head < 180)) || (head - newHead > 180)){
            LeftM.speed(0.5);
            RightM.speed(-0.5);
            head = getHead();
            //printf("heading: %f\n\r", head);
            wait(0.01);
        }
        else if ((head > newHead && (head-newHead < 180)) || (newHead - head > 180)){
            LeftM.speed(-0.5);
            RightM.speed(0.5);
            head = getHead();
            //printf("heading: %f\n\r", head);
            wait(0.01);
        }
    }
    LeftM.speed(0);
    RightM.speed(0);
    led2 = 0;
    countl = 0;
    countr = 0;
    wait(0.5);
}


void coord(int x, int y){
    if (y>yy)
        move((y-yy)*250,1);
    else if (y<yy)
        move((yy-y)*250,-1);
        
    if (x>xx){
        turn(1);
        move((x-xx)*250,1);
        turn(-1);
    }
    else if (x<xx){
        turn(-1);
        move((xx-x)*250,1);
        turn(1);
    }
    
    xx=x;
    yy=y;
}

int main()
{
    lhes.mode(PullUp);
    rhes.mode(PullUp);
    lhes.rise(&leftM_count);
    rhes.rise(&rightM_count);
    xx=0;
    yy=0;
    IMU.begin();
    if (!IMU.begin()) {
        led1 = 1;
        led2 = 1;
        led3 = 1;
    }
    IMU.calibrate(1);
    
    led3 = 1;
    IMU.calibrateMag(0);
    led3 = 0;
    wait(2);
    
//    while (1){
//        pc.printf("Heading: %f\n\r", getHead());
//        wait (1);
//    }
    
    coord(0,2);
    coord(2,2);
    coord(2,0);
    coord(0,0);
    
    speaker.period(1.0/note3);
    speaker =0.25;
    wait(1);
    speaker=0.0;
}

