/*****************************************************************************
*   @file    AD7172.h
*   @brief   AD7172 header file.
*   @devices AD7172-2
*
*   @author  
*
* 		Mauricio Donatti - mauricio.donatti@lnls.br
*		Lucas Tanio - lucas.tanio@lnls.br
*
* 		LNLS - Brazilian Synchrotron Light Source
* 		GIE - Electronics Instrumentation Group
*
* 		2020, April
*
*	Future Implementation:
*			-CRC and Checksum Support
*
******************************************************************************/

/*****************************************************************************
* Based on Analog Devices AD717X library, focused on performance improvements
*
* Special thanks to original authors:
*		acozma (andrei.cozma@analog.com)
*		dnechita (dan.nechita@analog.com)
* 
* Copyright 2015(c) Analog Devices, Inc.
*
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*  - Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*  - Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in
*    the documentation and/or other materials provided with the
*    distribution.
*  - Neither the name of Analog Devices, Inc. nor the names of its
*    contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*  - The use of this software may or may not infringe the patent rights
*    of one or more patent holders.  This license does not release you
*    from the requirement that you obtain separate licenses from these
*    patent holders to use this software.
*  - Use of the software either in source or binary form, must be run
*    on or directly connected to an Analog Devices Inc. component.
*
* THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR IMPLIED
* WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT, MERCHANTABILITY
* AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
* SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
* INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
* ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
* (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
******************************************************************************/

#ifndef AD7172_H
#define AD7172_H

#include "mbed.h"

#ifdef DEBUG

extern Serial pc;

//Config enabled DEBUG
#define AD7172_PRINTF(fmt, ...)       pc.printf("AD7172: " fmt "\r\n", ##__VA_ARGS__);
#else
#define AD7172_PRINTF(fmt, ...)         __NOP()
#endif

//Creating 32 bits variable
typedef union Data32
{
    uint32_t data;
    uint8_t bytes[4];    
}data32;

//Creating 16 bits variable
typedef union Data16
{
    uint16_t data;
    uint8_t bytes[2];     
}data16;

//AD7172-2 Register Map
#define AD7172_COMM_REG       0x00 //Comunication register address
#define AD7172_STATUS_REG     0x00 //Status register address
#define AD7172_ADCMODE_REG    0x01 //ADCMODE register address
#define AD7172_IFMODE_REG     0x02 //IFMODE register address
#define AD7172_REGCHECK_REG   0x03 //Register check address
#define AD7172_DATA_REG       0x04 //Data register address
#define AD7172_GPIOCON_REG    0x06 //GPIO communication register address
#define AD7172_ID_REG         0x07 //ID register address
#define AD7172_CHMAP0_REG     0x10 //CHx register address
#define AD7172_CHMAP1_REG     0x11
#define AD7172_CHMAP2_REG     0x12
#define AD7172_CHMAP3_REG     0x13
#define AD7172_SETUPCON0_REG  0x20 //SETUPCONx register address
#define AD7172_SETUPCON1_REG  0x21
#define AD7172_SETUPCON2_REG  0x22
#define AD7172_SETUPCON3_REG  0x23
#define AD7172_FILTCON0_REG   0x28 //FILTCONx register address
#define AD7172_FILTCON1_REG   0x29
#define AD7172_FILTCON2_REG   0x2A
#define AD7172_FILTCON3_REG   0x2B
#define AD7172_OFFSET0_REG    0x30 //OFFSETx register address
#define AD7172_OFFSET1_REG    0x31
#define AD7172_OFFSET2_REG    0x32
#define AD7172_OFFSET3_REG    0x33
#define AD7172_GAIN0_REG      0x38 //GAINx register address
#define AD7172_GAIN1_REG      0x39
#define AD7172_GAIN2_REG      0x3A
#define AD7172_GAIN3_REG      0x3B

//Communication Register bits
#define AD7172_COMM_REG_RD     (1 << 6) //R/W' - This bit determines if the command is a read or write operation

//Status Register bits
#define AD7172_STATUS_REG_RDY      (1 << 7) //RDY' -  This bit goes low when the ADC has written a new result to the data register
#define AD7172_STATUS_REG_ADC_ERR  (1 << 6)
#define AD7172_STATUS_REG_CRC_ERR  (1 << 5)
#define AD7172_STATUS_REG_REG_ERR  (1 << 4)
#define AD7172_STATUS_REG_CH(x)    ((x) & 0x03)

//ADC Mode Register bits
#define AD7172_ADCMODE_REG_REF_EN     	(1 << 15) //Enables internal reference
#define AD7172_ADCMODE_REG_HIDE_DELAY   (1 << 14)
#define AD7172_ADCMODE_SING_CYC       	(1 << 13)
#define AD7172_ADCMODE_REG_DELAY(x)   	(((x) & 0x7) << 8)
#define AD7172_ADCMODE_REG_MODE(x)    	(((x) & 0x7) << 4)
#define AD7172_ADCMODE_REG_CLKSEL(x) 	(((x) & 0x3) << 2) //Clock selection

// Interface Mode Register bits
#define AD7172_IFMODE_REG_ALT_SYNC      (1 << 12)
#define AD7172_IFMODE_REG_IOSTRENGTH    (1 << 11)
#define AD7172_IFMODE_REG_HIDE_DELAY    (1 << 10)
#define AD7172_IFMODE_REG_DOUT_RESET    (1 << 8)
#define AD7172_IFMODE_REG_CONT_READ     (1 << 7) //Enables a continuous read of the ADC data register
#define AD7172_IFMODE_REG_DATA_STAT     (1 << 6) //Append the status register to data register
#define AD7172_IFMODE_REG_REG_CHECK     (1 << 5)
#define AD7172_IFMODE_REG_XOR_EN        (0x01 << 2)
#define AD7172_IFMODE_REG_CRC_EN        (0x02 << 2)
#define AD7172_IFMODE_REG_XOR_STAT(x)   (((x) & AD7172_IFMODE_REG_XOR_EN) == AD7172_IFMODE_REG_XOR_EN)
#define AD7172_IFMODE_REG_CRC_STAT(x)   (((x) & AD7172_IFMODE_REG_CRC_EN) == AD7172_IFMODE_REG_CRC_EN)
#define AD7172_IFMODE_REG_DATA_WL16     (1 << 0)

//GPIO Configuration Register bits
#define AD7172_GPIOCON_REG_MUX_IO      (1 << 12)
#define AD7172_GPIOCON_REG_SYNC_EN     (1 << 11)
#define AD7172_GPIOCON_REG_ERR_EN(x)   (((x) & 0x3) << 9)
#define AD7172_GPIOCON_REG_ERR_DAT     (1 << 8)
#define AD7172_GPIOCON_REG_IP_EN1      (1 << 5)
#define AD7172_GPIOCON_REG_IP_EN0      (1 << 4)
#define AD7172_GPIOCON_REG_OP_EN1      (1 << 3)
#define AD7172_GPIOCON_REG_OP_EN0      (1 << 2)
#define AD7172_GPIOCON_REG_DATA1       (1 << 1)
#define AD7172_GPIOCON_REG_DATA0       (1 << 0)

//Channel Map Register 0-3 bits
#define AD7172_CHMAP_REG_CH_EN         (1 << 15)
#define AD7172_CHMAP_REG_SETUP_SEL(x)  (((x) & 0x3) << 12)
#define AD7172_CHMAP_REG_AINPOS(x)     (((x) & 0x1F) << 5)
#define AD7172_CHMAP_REG_AINNEG(x)     (((x) & 0x1F) << 0)

//Setup Configuration Register 0-3 bits
#define AD7172_SETUP_CONF_REG_BI_UNIPOLAR  	(1 << 12) //This bit sets the output coding of the ADC - 0-Unipolar, 1-Bipolar
#define AD7172_SETUP_CONF_REG_REFBUF_P    	(1 << 11)
#define AD7172_SETUP_CONF_REG_REFBUF_N    	(1 << 10)
#define AD7172_SETUP_CONF_REG_AINBUF_P    	(1 << 9)
#define AD7172_SETUP_CONF_REG_AINBUF_N    	(1 << 8)
#define AD7172_SETUP_CONF_REG_REF_SEL(x)   	(((x) & 0x3) << 4) //Reference selection

//Filter Configuration Register 0-3 bits
#define AD7172_FILT_CONF_REG_SINC3_MAP    (1 << 15)
#define AD7172_FILT_CONF_REG_ENHFILTEN    (1 << 11) 
#define AD7172_FILT_CONF_REG_ENHFILT(x)   (((x) & 0x7) << 8) //Postfilters for enhanced 50 Hz and 60 Hz rejection
#define AD7172_FILT_CONF_REG_ORDER(x)     (((x) & 0x3) << 5) //Digital filter 
#define AD7172_FILT_CONF_REG_ODR(x)       (((x) & 0x1F) << 0) //Sample rate

//Class Declaration
class AD7172
{
public:
	//Class Constructor
	AD7172(SPI& p_spi,PinName slave_select,DigitalIn& p_rdy);
	
	//Enable Device - CS goes low
	void enable();
		
	//Disable Device - CS goes high
	void disable();
	
	//Start Continuous Convertion Mode
	void start_continuous();
	
	//Stop Continuous Convertion Mode
	void stop_continuous();
			
	//Read ID function (update ID variable) - Communication Test
	void ReadID();
	
	//Read Status register and update status and ready public variables
	void ReadStatus();

	//Reads the value of the specified register.
	void ReadRegister(uint8_t reg,uint8_t bytes);
	
	//Writes the value of the specified register.
	void WriteRegister(uint8_t reg,uint8_t bytes);
	
	//Resets the device
	void Reset();
	
	//Waits until a new conversion result is available
	void WaitForReady(uint32_t timeout);
	
	//Reads the conversion result from the device
	void ReadDataRegister();
	
	//Reads the conversion result from the device with status reg appended
	void ReadDataRegisterStatus();
	
	//Reads data for continuous read mode
	void ReadDataContinuous();
	
	//Reads data for continuous read mode if DATA STAT bit is set
	void ReadDataContinuousStatus();
	
    uint8_t status;
    uint8_t sw_ready;
    uint8_t bytes;
    uint8_t channel;
    uint8_t addr;
    
    uint8_t res;
    
	data32 data; //Creating a variable named data, type data32
	data16 id; //Creating a variable named id, type data16
	DigitalIn &_rdy;

protected:
    SPI &_spi;
	DigitalOut* cs;
		
private:
    int i;
	uint8_t aux;
	uint8_t continuous_on;
};

#endif //AD7172_H
