/**
@file main.h
@brief Header file containing functions prototypes and global variables.
@brief Implementation of a Pedometer using the accelerometer MMA8452Q, Nokia 5110 display and the mbed LPC1768.
@brief Revision 1.5.
@author Edson Manoel da Silva
@date   May 2015
*/
#ifndef MAIN_H
#define MAIN_H

#include "mbed.h"
#include "MMA8452.h"
#include "N5110.h"
#include "PowerControl/EthernetPowerControl.h" 


//   
/**  
@namespace lcd
@brief GPIO output for Nokia 5110 Display
@brief  VCC (p7),SCE(p8),RST(p9),D/C(p10),MOSI(p11),SCLK(p13),LED(p26)
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);

/**  
@namespace mma8452
@brief GPIO output for mma8452 accelerometer sensor
@brief SDA(p28), SCL(p27)
*/
MMA8452 mma8452(p28,p27);

/**  
@namespace leds
@brief Debug leds
*/
BusOut leds(LED4,LED3,LED2,LED1);

/**  
@namespace buzzer
@brief PWM for the buzzer
*/
PwmOut buzzer(p21);

/**  
@namespace serial
@brief Serial interface
*/
Serial serial(USBTX,USBRX);

/**  
@namespace timer1
@brief First timer used
*/
Timeout timer1;

/**  
@namespace timer2
@brief Second timer used
*/
Timeout timer2;

/**  
@namespace timer3
@brief Chronometer timer
*/
Ticker timer3;

/**  
@namespace local
@brief Create Local File System
*/
LocalFileSystem local("local"); // create local filesystem

/**  
@namespace I1
@brief Create the interrupt for Transient Detection
@brief I1(p15)
*/
InterruptIn I1(p15);

/**  
@namespace I2
@brief Create the interrupt for Pulse(Tap) Detection
@brief I2(p16)
*/
InterruptIn I2(p16);

/**  
@namespace mma8452
@brief Accleration structure declared in MMA8452 class
*/
Acceleration acceleration;  

/**  
@namespace mma8452
@brief Accleration average structure declared in MMA8452 class
*/
Acceleration acc_avg;

unsigned char second = 0; /*!< second flag set in TimerExpired3 */
unsigned char minute = 0; /*!< minute flag set in TimerExpired3 */
unsigned char hour = 0;/*!< hour flag set in TimerExpired3 */
unsigned char state = 0;/*!< state variable for the FSM */
unsigned char I1_flag = 0;/*!< Interrupt flag set in Transient Detection Interrupt */
unsigned char I2_flag = 0;/*!< Interrupt flag set in Pulse(Tap) Detection Interrupt */
unsigned char timerFlag1 = 0;/*!< Interrupt flag set in Timer1 */
unsigned char timerFlag2 = 0;/*!< Interrupt flag set in Timer2 */
unsigned char aux=0;/*!< Auxiliar for checking if the user stopped using the device */

char Int_SourceSystem =0;/*!< Variable used to read the MMA8452Q Interrupt Source Register */
char Int_SourceTrans=0;/*!< Variable used to clear the MMA8452Q Interrupt Registers */

unsigned char length;/*!< Variable used to check the string length to be printed in the LCD */
char buffer[14];/*!< Buffer used for printing strings on the display */

unsigned int step = 0;/*!< Variable used to ccalculate the steps */
float km = 0;/*!< Variable used to ccalculate the kilometers */
//float km_day[30] = {0,2,3,4,5,6,7,8,1,2,3,4,5,6,7,8,1,2,3,4,5,6,7,8,1,2,3,4,5,6};
float km_day[30] = {};/*!< Variable used to ccalculate the kilometers per day */

float acc_vector = 0;/*!< Variable for check if a step was performed */
double sub_x = 0;/*!< Variable used for calibration */
double sub_y = 0;/*!< Variable used for calibration */
double sub_z = 0;/*!< Variable used for calibration */

/**
Set a flag to alert that a Transient Detection Interrupt has ocurred
*/
void Interrupt();

/**
Set a flag to alert that a Pulse Detection Interrupt has ocurred
*/
void Interrupt2();

/**
Blind the LEDS for state machine error alert
*/
void error();

/**
Set a flag to alert that a Timer1 Interrupt has ocurred
*/
void TimerExpired1();

/**
Set a flag to alert that a Timer2 Interrupt has ocurred
*/
void TimerExpired2();

/**
Performs the calculation for the chronometer time
*/
void TimerExpired3();

/**
Saves the data collected in the stepping count to the flash disk
@param date - the date of the data
@param data1 - steps
@param data2 - Kilometer
*/
void writeDataToFile(char *date,int data1,float data2);
#endif