/**
@file MMA8452.cpp

@brief Member functions implementations
@see https://developer.mbed.org/users/eencae/code/MMA8452/
*/
#include "mbed.h"
#include "MMA8452.h"

MMA8452:: MMA8452(PinName sdaPin, PinName sclPin)
{
    i2c = new I2C(sdaPin,sclPin); // create new I2C instance and initialise
    i2c->frequency(400000);       // I2C Fast Mode - 400kHz
    leds = new BusOut(LED4,LED3,LED2,LED1);  // for debug
}

void MMA8452::init()
{

    i2c->frequency(400000); // set Fast Mode I2C frequency (5.10 datasheet)

    char data = readByteFromRegister(WHO_AM_I);  // p18 datasheet
    if (data != 0x2A) { // if correct ID not found, hand and flash error message
        error();
    }

    // put into STANDBY while configuring
    data = readByteFromRegister(CTRL_REG1); // get current value of register
    data &= ~(1<<0); // clear bit 0 (p37 datasheet)
    sendByteToRegister(data,CTRL_REG1);

    // Set output data rate, default is 800 Hz, will set to 100 Hz (clear b5, set b4/b3 - p37 datasheet)
    data = readByteFromRegister(CTRL_REG1);
    data &= ~(1<<5);
    data |=  (1<<4);
    data |=  (1<<3);
    sendByteToRegister(data,CTRL_REG1);
    
    //Interrupt Control Register (Set b1 - Interrupt polarity ACTIVE high - p39 Datasheet)
    data = readByteFromRegister(CTRL_REG3);
    data |=  (1<<1); //IPOL bit selects the polarity of the interrupt signal. 
                    //When IPOL is ‘0’ (default value) any interrupt event will signaled with a logical 0.
    sendByteToRegister(data,CTRL_REG3);

    //// Can also change default 2g range to 4g or 8g (p22 datasheet)
    data = readByteFromRegister(XYZ_DATA_CFG);
    data |=  (1<<0); // set bit 0 - 4g range
    sendByteToRegister(data,XYZ_DATA_CFG);

    //Configuring the Transient Configuration Register
     // set bit 1 and 2 - This enables the X-axis and the Y-axis transient detection
    sendByteToRegister(0x16,TRANSIENT_CFG);
    
     //Configuring the Pulse Configuration Register
     // set bit 2 and 6 - This enables the Y-axis single pulse detection
    sendByteToRegister(0x44,PULSE_CFG);
    
    //Configuring Transient Threshold Register
    //  This set the X-axis and the Y-axis transient threshold to X or Y > 2.5g
    //Note: Step count is 0.063g per count
    //2.0g / 0.063g = 31.74. Therefore set the threshold to 32 counts
    sendByteToRegister(0x20,TRANSIENT_THS);
    
    //Configuring Pulse Threshold Register
    //  This set the Y-axis transient threshold to  Y > 0.5g
    //Note: Step count is 0.063g per count
    //0.5/0.063g = 7.93 counts. Therefore set the threshold to 8 counts
    sendByteToRegister(0x08,PULSE_THSY);
    
    //Set the debounce counter to 100ms
    //Note: 100 Hz ODR, therefore 10 ms step sizes
    sendByteToRegister(0x0A,TRANSIENT_COUNT);
    
    //Set the Time Limit for Tap Detection to 100ms
    //Note: 100 Hz ODR, therefore 2.5 ms step sizes
    sendByteToRegister(0x28,PULSE_TMLT);
    
    //Set the Latency Time to 300ms
    //Note: 100 Hz ODR, therefore 5 ms step sizes
    sendByteToRegister(0x3C,PULSE_LTCY);

    //Enable the Transient Detection (Bit 5) and Pulse Detection (Bit 3)
    //  Enable the Interrupt in Register 0x2D
    sendByteToRegister(0x28,CTRL_REG4);
   
    //  To set the Transient function to INT 1, set bit 5 in register 0x2E.
    //  To set the Transient function to INT 2, clear bit 3 in register 0x2E.
    sendByteToRegister(0x20,CTRL_REG5);    
    
    // set ACTIVE
    data = readByteFromRegister(CTRL_REG1);
    data |= (1<<0);   // set bit 0 in CTRL_REG1
    sendByteToRegister(data,CTRL_REG1);

}

void MMA8452::transient_counting() //configuring the transient threshold for counting steps
{

    i2c->frequency(400000); // set Fast Mode I2C frequency (5.10 datasheet)

    char data = readByteFromRegister(WHO_AM_I);  // p18 datasheet
    if (data != 0x2A) { // if correct ID not found, hand and flash error message
        error();
    }

    // put into STANDBY while configuring
    data = readByteFromRegister(CTRL_REG1); // get current value of register
    data &= ~(1<<0); // clear bit 0 (p37 datasheet)
    sendByteToRegister(data,CTRL_REG1);

    // Set output data rate 800Hz
    data = readByteFromRegister(CTRL_REG1);
    data &= ~(1<<5);
    data &= ~(1<<4);
    data &= ~(1<<3);
    sendByteToRegister(data,CTRL_REG1);
    

    // Set the high-pass filter cutoff frequency for removal of the offset and slower changing acceleration data
    sendByteToRegister(0x00,HP_FILTER_CUTOFF); //16Hz

    //// Can also change default 2g range to 4g or 8g (p22 datasheet)
    data = readByteFromRegister(XYZ_DATA_CFG);
//    data |=  (1<<0); // set bit 0 - 4g range
    data |=  (1<<4); // set bit 4 - The data registers 0x01 - 0x06 will contain high-pass filtered data when this bit is set.
    sendByteToRegister(data,XYZ_DATA_CFG);
    
    
    // set ACTIVE
    data = readByteFromRegister(CTRL_REG1);
    data |= (1<<0);   // set bit 0 in CTRL_REG1
    sendByteToRegister(data,CTRL_REG1);

}

// read acceleration data from device
Acceleration MMA8452::readValues()
{
    // acceleration data stored in 6 registers (0x01 to 0x06)
    // device automatically increments register, so can read 6 bytes starting from OUT_X_MSB
    char data[6];
    readBytesFromRegister(OUT_X_MSB,6,data);

    char x_MSB = data[0];  // extract MSB and LSBs for x,y,z values
    char x_LSB = data[1];
    char y_MSB = data[2];
    char y_LSB = data[3];
    char z_MSB = data[4];
    char z_LSB = data[5];

    // [0:7] of MSB are 8 MSB of 12-bit value , [7:4] of LSB are 4 LSB's of 12-bit value
    // need to type-cast as numbers are in signed (2's complement) form (p20 datasheet)
    int x = (int16_t) (x_MSB << 8) | x_LSB;  // combine bytes
    x >>= 4;  // are left-aligned, so shift 4 places right to right-align
    int y = (int16_t) (y_MSB << 8) | y_LSB;
    y >>= 4;
    int z = (int16_t) (z_MSB << 8) | z_LSB;
    z >>= 4;

    // sensitivity is 1024 counts/g in 2g mode (pg 9 datasheet)
    //  "   "          512      "      4g     "
    //  "   "          256      "      8g     "
    Acceleration acc;
    
    acc.x = x/512.0;
    acc.y = y/512.0;
    acc.z = z/512.0;
    
    return acc;
}


// read 50 times a acceleration data from device and make the average
Acceleration MMA8452::average()
{
    int i = 0;
    int x_avg = 0;
    int y_avg = 0;
    int z_avg = 0;
        
    while (i < 50)
    {
        // acceleration data stored in 6 registers (0x01 to 0x06)
        // device automatically increments register, so can read 6 bytes starting from OUT_X_MSB
        char data[6];
        readBytesFromRegister(OUT_X_MSB,6,data);
    
        char x_MSB = data[0];  // extract MSB and LSBs for x,y,z values
        char x_LSB = data[1];
        char y_MSB = data[2];
        char y_LSB = data[3];
        char z_MSB = data[4];
        char z_LSB = data[5];
    
        // [0:7] of MSB are 8 MSB of 12-bit value , [7:4] of LSB are 4 LSB's of 12-bit value
        // need to type-cast as numbers are in signed (2's complement) form (p20 datasheet)
        int x = (int16_t) (x_MSB << 8) | x_LSB;  // combine bytes
        x >>= 4;  // are left-aligned, so shift 4 places right to right-align
        int y = (int16_t) (y_MSB << 8) | y_LSB;
        y >>= 4;
        int z = (int16_t) (z_MSB << 8) | z_LSB;
        z >>= 4;
        
        x_avg = x_avg + x;
        y_avg = y_avg + y;
        z_avg = z_avg + z;
        
        i = i + 1;
        wait(0.05);
    }
    
    // sensitivity is 1024 counts/g in 2g mode (pg 9 datasheet)
    //  "   "          512      "      4g     "
    //  "   "          256      "      8g     "
    Acceleration acc_avg;
    
    acc_avg.x = x_avg/25600.0; //512 * 50 //50 comes from the average of 50 readings
    acc_avg.y = y_avg/25600.0;
    acc_avg.z = z_avg/25600.0;
    
    return acc_avg;
}

// reads a byte from a specific register
char MMA8452::readByteFromRegister(char reg)
{
    int nack = i2c->write(MMA8452_W_ADDRESS,&reg,1,true);  // send the register address to the slave
    // true as need to send repeated start condition (5.10.1 datasheet)
    // http://www.i2c-bus.org/repeated-start-condition/
    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

    char rx;
    nack = i2c->read(MMA8452_R_ADDRESS,&rx,1);  // read a byte from the register and store in buffer
    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

    return rx;
}

// reads a series of bytes, starting from a specific register
void MMA8452::readBytesFromRegister(char reg,int numberOfBytes,char bytes[])
{

    int nack = i2c->write(MMA8452_W_ADDRESS,&reg,1,true);  // send the slave write address and the configuration register address
    // true as need to send repeated start condition (5.10.1 datasheet)
    // http://www.i2c-bus.org/repeated-start-condition/

    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

    nack = i2c->read(MMA8452_R_ADDRESS,bytes,numberOfBytes);  // read bytes
    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

}

// sends a byte to a specific register
void MMA8452::sendByteToRegister(char byte,char reg)
{
    char data[2];
    data[0] = reg;
    data[1] = byte;
    // send the register address, followed by the data
    int nack = i2c->write(MMA8452_W_ADDRESS,data,2);
    if (nack)
        error();  // if we don't receive acknowledgement, flash error message

}

void MMA8452::error()
{
    while(1) {
        leds->write(15);
        wait(0.1);
        leds->write(0);
        wait(0.1);
    }
}