/**
 * @file
 * @section LICENSE
 * Copyright (c) 2010 Mustafa Ozgur Kanli.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section Description
 * Implementation of MaxSonar class.
 */

#include "mbed.h"
#include "DimEngAcc3D.h"

DimEngAcc3D::DimEngAcc3D(PinName pin_x, PinName pin_y, PinName pin_z) {

    // Set defaults.
    initFromVoltage(3.3);
    this->ain_x = NULL;
    this->ain_y = NULL;
    this->ain_z = NULL;

    // Create analog inputs as required.
    if (pin_x != NC)
        this->ain_x = new AnalogIn(pin_x);

    if (pin_y != NC)
        this->ain_y = new AnalogIn(pin_y);

    if (pin_z != NC)
        this->ain_z = new AnalogIn(pin_z);

}

DimEngAcc3D::~DimEngAcc3D(void) {

    if (this->ain_x != NULL)
        delete this->ain_x;

    if (this->ain_y != NULL)
        delete this->ain_y;

    if (this->ain_z != NULL)
        delete this->ain_z;
}

void DimEngAcc3D::initFromVoltage(float voltage) {
    /*
     * The equation was obtained from the linear relationship
     * of the sensitivity and input/reference voltage and two
     * pairs of sensitivity & voltage value pairs:
     * 195 mV/g @ 2V and 360mV @ 3.6V
     *
     * s = sensitivity (V/g)
     * v = reference voltage (V)
     * s = 0.103125*v - 0.01125
     */
    this->ref_voltage = voltage;
    this->zero_point = this->ref_voltage / 2.0;
    this->sensitivity = (0.103125 * this->ref_voltage) - 0.01125;
    this->x = this->zero_point;
    this->y = this->zero_point;
    this->z = this->zero_point;
    
    printf("ref: %f, zero: %f, sens: %f \r\n", this->ref_voltage, this->zero_point, this->sensitivity);
}

void DimEngAcc3D::setVoltage(float voltage) {
    initFromVoltage(voltage);
}

void DimEngAcc3D::read(void) {

    // Read the acceleration outputs.
    if (this->ain_x != NULL)
        this->x = this->ain_x->read() * this->ref_voltage;

    if (this->ain_y != NULL)
        this->y = this->ain_y->read() * this->ref_voltage;

    if (this->ain_z != NULL)
        this->z = this->ain_z->read() * this->ref_voltage;

    // Convert raw voltage measurements to m/s/s
    this->x = ACC_G * (this->x - this->zero_point) / this->sensitivity;
    this->y = ACC_G * (this->y - this->zero_point) / this->sensitivity;
    this->z = ACC_G * (this->z - this->zero_point) / this->sensitivity;
}

