/**
 * @file
 * @section LICENSE
 * Copyright (c) 2010 Mustafa Ozgur Kanli.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"),to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @section Description
 * This class provides an object to obtain 3D acceleration measurements from a
 * Dimension Engineering Buffered +-3g Tri-axis Accelerometer. This device uses
 * the ADXL330. Measurements are obtained using 3 analog inputs.
 *
 */
#ifndef DIMENG_ACC3D_H_
#define DIMENG_ACC3D_H_

#include "mbed.h"

/**
 * Acceleration due to gravity at sea level on Earth (m/s/s).
 */
#define ACC_G	9.80665

/**
 * Class to read acceleration measurements from Dimension Engineering
 * Buffered +-3g Tri-axis Accelerometer device.
 *
 * Example
 * @code
 * #include "mbed.h"
 * #include "DimEngAcc3D.h"
 *
 * int main() {
 *     DimEngAcc3D *acc;
 *
 *     // Create and configure object for 3.3V powered device,
 *     acc = new DimEngAcc3D(p18, p19, p20);
 *     acc->setVoltage(3.3);
 *
 *     while(1) {
 *         // Read.
 *         acc->read();
 *
 *         // Print and delay 0.5s.
 *         printf("%.3f, %.3f, %.3f\r\n", acc->x, acc->y, acc->z);
 *         wait(0.5);
 *     }
 * }
 * @endcode
 */
class DimEngAcc3D {

private:
    float ref_voltage;          //!< Supply/reference voltage (V).
    float zero_point;			//!< Zero g voltage output (V).
    float sensitivity;          //!< Sensitivity at reference voltage (V/g).
    AnalogIn *ain_x;            //!< For analog reads of acceleration in x axis.
    AnalogIn *ain_y;            //!< For analog reads of acceleration in y axis.
    AnalogIn *ain_z;            //!< For analog reads of acceleration in z axis.

    /**
     * Initialize private variables according to the specified
     * reference voltage
     * @param   voltage The specified voltage.
     */
    void initFromVoltage(float voltage);

public:
    float x;					//!< Most recent acceleration in x axis (m/s/s).
    float y;					//!< Most recent acceleration in y axis (m/s/s).
    float z;					//!< Most recent acceleration in z axis (m/s/s).

    /**
     * Constructor.
     *
     * @param   pin_x    Pin connected to output x of device.
     * @param   pin_y    Pin connected to output y of device.
    * @param   pin_z    Pin connected to output z of device.
     * @note    Unused/unconnected outputs may be specified as NC.
     */
    DimEngAcc3D(PinName pin_x, PinName pin_y, PinName pin_z);

    /**
     * Destructor.
     */
    ~DimEngAcc3D(void);

    /**
     * Specify the supply voltage used by the device.
     *
     * @param   voltage The specified voltage (default 3.3)
     * @note    This is important for correct conversion of the voltage
     *          from output pins of device into acceleration in m/s/s.
     */
    void setVoltage(float voltage);

    /**
     * Read acceleration and store in x, y & z.
     */
    void read(void);
};

#endif
