// TLC5940 Tester
//
// This is a simple test bed for the TLC5940 chips.  We use the same
// interface class used in the Pinscape v2 firmware, but omit all of the
// other functionality of the Pinscape software.  This does nothing
// besides drive the TLC5940 chips.
//
// The GPIO pin connections to the TLC5940 chips are configured by default 
// for the Pinscape Expansion Boards.  Edit the "TLC5940 configuration" 
// section below to change the pin assignments.

#include "mbed.h"
#include "TLC5940.h"

// Operation mode:
//
//   0 = ALL ON mode - all outputs continuously turned on
//
//   1 = Up/Down mode - all outputs ramped from off to full brightness
//       and back every couple of seconds
//
//   2 = Sweep mode - outputs ramped up one at a time then ramped 
//       down one at a time
//
const int MODE = 0;           

// TLC5940 configuration
const int NCHIPS    = 2;      // number of TLC5940 chips in daisy chain
const PinName SCLK  = PTC5;
const PinName SIN   = PTC6;
const PinName GSCLK = PTA1;
const PinName BLANK = PTC7;
const PinName XLAT  = PTC10;


// diagnostic LED segments
DigitalOut ledR(LED1), ledG(LED2), ledB(LED3);

// adjust an output value by an increment, keeping it within range
static void adjustOut(int &out, int inc)
{
    out += inc;
    if (out < 0)
        out = 0;
    else if (out > 4095)
        out = 4095;
}

int main() 
{
    // set up the tlc5940 interface
    TLC5940 tlc5940(SCLK, SIN, GSCLK, BLANK, XLAT, NCHIPS);
    
    // start the tlc5940 clock
    tlc5940.start();
    
    // enable the outputs
    tlc5940.enable(true);
    
    // current status for ramp mode
    struct
    {
        int dir;
        int val;
    } ramp = { 1, 0 };
    
    // current status for sweep mode
    struct
    {
        int out;
        int dir;
        int val;
    } sweep = { 0, 4, 0 };
     
    // in MODE 0, set all outputs to full on initially
    if (MODE == 0)
    {
        for (int i = 0 ; i < NCHIPS*16 ; ++i)
            tlc5940.set(i, 4095);
    }
    
    // heartbeat flasher
    int t = 0;
    int hb = 0;

    // send tlc updates forever
    for (;;)
    {
        // send TLC updates as needed
        tlc5940.send();
        
        // pause between operations
        wait_us(500);
        
        // update outputs according to the current mode
        switch (MODE)
        {
        case 1:
            // ramp up/down mode - all outputs ramp on every loop
            
            // switch direction if at the top or bottom of the ramp
            if ((ramp.dir > 0 && ramp.val == 4095) || (ramp.dir < 0 && ramp.val == 0))
                ramp.dir = -ramp.dir;
                
            // adjust all outputs
            adjustOut(ramp.val, ramp.dir);
            for (int i = 0 ; i < NCHIPS*16 ; ++i)
                tlc5940.set(i, ramp.val);
               
            // done 
            break;
            
        case 2:
            // sweep up/down mode - adjust one output at a time
            
            // at top or bottom, move to next output
            if ((sweep.dir > 0 && sweep.val == 4095) || (sweep.dir <= 0 && sweep.val == 0))
            {
                // advance to the next output
                if (++sweep.out < NCHIPS*16)
                {
                    // at next out - start at other end of ramp
                    sweep.val = (sweep.dir > 0 ? 0 : 4095);
                }
                else
                {
                    // past last output - wrap and reverse the ramp direction
                    sweep.out = 0;
                    sweep.dir = -sweep.dir;
                }
            }
            
            // adjust this output
            adjustOut(sweep.val, sweep.dir);
            tlc5940.set(sweep.out, sweep.val);
            
            // done
            break;
        }
        
        // update the heartbeat if needed
        if (t++ > 1000)
        {
            ledR = 1; ledB = 1; ledG = 1;
            switch (hb++)
            {
            case 0: ledR = 0; ledG = 0; break;  // yellow
            case 1: ledB = 0; ledR = 0; break;  // purple
            case 2: ledG = 0; break;            // green
            case 3: ledB = 0; break;            // blue
            }
            hb %= 3;
            t = 0;
        }
    }
}
