#include "mbed.h"
#include "Adafruit_PN532.h"

#define MOSI p11
#define MISO p12
#define SCK  p13
#define SS   p20

Serial pc(USBTX, USBRX);
Adafruit_PN532 nfc(SCK, MISO, MOSI, SS);

void loop(void);
int sectorToBlock(int sectorIndex);

int main()
{
    pc.printf("Hello v1.0.1!\r\n");
    nfc.begin();

    uint32_t versiondata = nfc.getFirmwareVersion();
    if (! versiondata) {
        pc.printf("Didn't find PN53x board");
        while (1); 
    }
    
    pc.printf("Found chip PN5%2X with Firmware ver. %d.%d\r\n", versiondata >> 24, (versiondata >> 16) & 0xFF, (versiondata >> 8) & 0xFF);
    nfc.SAMConfig();
    pc.printf("Waiting for an ISO14443A Card ...\r\n");

    while(1) {
        loop();
    }
}


void loop(void)
{
    uint8_t uid[] = { 0, 0, 0, 0, 0, 0, 0 };  // Buffer to store the returned UID
    uint8_t uidLength;                        // Length of the UID (4 or 7 bytes depending on ISO14443A card type)
    int sector = 32;

    if (nfc.readPassiveTargetID(PN532_MIFARE_ISO14443A, uid, &uidLength)) {
        // Display some basic information about the card
        pc.printf("\r\n\r\nFound an ISO14443A card\r\n");
        pc.printf("  UID Length: %d bytes", uidLength);
        pc.printf("  UID Value: ");
        nfc.PrintHex(uid, uidLength);
        pc.printf("\r\n");

        if (uidLength == 4) {
            pc.printf("Seems to be a Mifare Classic card (4 byte UID)\r\n");
            pc.printf("Trying to authenticate sector %d with default KEYA value\r\n", sector);
            uint8_t keya[6] = { 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5 };
            int firstBlockInSectorNumber = sectorToBlock(sector);
            
            uint8_t buffer[16 * 4];
            for(int blockOffset = 0; blockOffset < 4; blockOffset++){
                int blockNumber = firstBlockInSectorNumber + blockOffset;
                int bufferLoc = 16 * blockOffset;
                
                if (nfc.mifareclassic_AuthenticateBlock(uid, uidLength, blockNumber, 0, keya)) {
                    uint8_t data[16];
    
                    if (nfc.mifareclassic_ReadDataBlock(blockNumber, data)) {
                        // Data seems to have been read ... spit it out
                        pc.printf("Reading Block %d:\r\n\t", blockNumber);
                        memcpy(buffer + bufferLoc, data, 16);
    
                        // Wait a bit before reading the card again
                        wait(1);
                    } else {
                        pc.printf("Ooops ... unable to read the requested block.  Try another key?\r\n");
                    }
                } else {
                    pc.printf("Ooops ... authentication failed: Try another key?\r\n");
                }
            }
            printf("<FullRead>\r\n");
            Adafruit_PN532::PrintHexChar(buffer, 16 * 4);
            printf("</FullRead>\r\n");
        }
    }
}

int sectorToBlock(int sectorIndex)
{
    if (sectorIndex < 32) {
        return sectorIndex * 4;
    } else {
        return 32 * 4 + (sectorIndex - 32) * 16;
    }
}