/**
 * C328-7640 device driver class (Version 0.0.6)
 * Reference documents: C328-7640 User Manual v3.0 2004.8.19
 *
 * Copyright (C) 2010 Shinichiro Nakamura (CuBeatSystems)
 * http://shinta.main.jp/
 */

#include "mbed.h"
#include "SerialBuffered.h"

#ifndef _CAMERA_C328_H_
#define _CAMERA_C328_H_

/*
 * Class: CameraC328
 */
class CameraC328 {
public:

    /**
     * Color type.
     */
    enum ColorType {
        GrayScale2bit = 0x01,   // 2bit for Y only
        GrayScale4bit = 0x02,   // 4bit for Y only
        GrayScale8bit = 0x03,   // 8bit for Y only
        Color12bit = 0x05,      // 444 (RGB)
        Color16bit = 0x06,      // 565 (RGB)
        Jpeg = 0x07
    };

    /**
     * Raw resolution.
     */
    enum RawResolution {
        RawResolution80x60 = 0x01,
        RawResolution160x120 = 0x03
    };

    /**
     * JPEG resolution.
     */
    enum JpegResolution {
        JpegResolution80x64 = 0x01,
        JpegResolution160x128 = 0x03,
        JpegResolution320x240 = 0x05,
        JpegResolution640x480 = 0x07
    };

    /**
     * Error number.
     */
    enum ErrorNumber {
        NoError = 0x00,
        PictureTypeError = 0x01,
        PictureUpScale = 0x02,
        PictureScaleError = 0x03,
        UnexpectedReply = 0x04,
        SendPictureTimeout = 0x05,
        UnexpectedCommand = 0x06,
        SramJpegTypeError = 0x07,
        SramJpegSizeError = 0x08,
        PictureFormatError = 0x09,
        PictureSizeError = 0x0a,
        ParameterError = 0x0b,
        SendRegisterTimeout = 0x0c,
        CommandIdError = 0x0d,
        PictureNotReady = 0x0f,
        TransferPackageNumberError = 0x10,
        SetTransferPackageSizeWrong = 0x11,
        CommandHeaderError = 0xf0,
        CommandLengthError = 0xf1,
        SendPictureError = 0xf5,
        SendCommandError = 0xff
    };

    /**
     * Picture type.
     */
    enum PictureType {
        SnapshotPicture = 0x01,
        PreviewPicture = 0x02,
        JpegPreviewPicture = 0x05
    };

    /**
     * Snapshot type.
     */
    enum SnapshotType {
        CompressedPicture = 0x00,
        UncompressedPicture = 0x01
    };

    /**
     * Baud rate.
     */
    enum Baud {
        Baud7200 = 7200,
        Baud9600 = 9600,
        Baud14400 = 14400,
        Baud19200 = 19200,  // Default.
        Baud28800 = 28800,
        Baud38400 = 38400,
        Baud57600 = 57600,
        Baud115200 = 115200
    };

    /**
     * Reset type.
     */
    enum ResetType {
        ResetWholeSystem = 0x00,
        ResetStateMachines = 0x01
    };

    /**
     * Data type.
     */
    enum DataType {
        DataTypeSnapshotPicture = 0x01,
        DataTypePreviewPicture = 0x02,
        DataTypeJpegPreviewPicture = 0x05
    };

    /**
     * Constructor.
     *
     * @param tx A pin for transmit.
     * @param rx A pin for receive.
     * @param baud Baud rate. (Default is Baud19200.)
     */
    CameraC328(PinName tx, PinName rx, Baud baud =  Baud115200);

    /**
     * Destructor.
     */
    ~CameraC328();

    /**
     * Make a sync. for baud rate.
     */
    ErrorNumber sync();

    /**
     * Initialize.
     *
     * @param ct Color type.
     * @param rr Raw resolution.
     * @param jr JPEG resolution.
     */
    ErrorNumber init(ColorType ct, RawResolution rr, JpegResolution jr);

    /**
     * Get uncompressed snapshot picture.
     *
     * @param func A pointer to a callback function.
     *             Please do NOT block this callback function.
     *             Because the camera module transmit image datas continuously.
     * @return Status of the error.
     */
    ErrorNumber getUncompressedSnapshotPicture(void(*func)(size_t done, size_t total, char c));

    /**
     * Get uncompressed preview picture.
     *
     * @param func A pointer to a callback function.
     *             Please do NOT block this callback function.
     *             Because the camera module transmit image datas continuously.
     * @return Status of the error.
     */
    ErrorNumber getUncompressedPreviewPicture(void(*func)(size_t done, size_t total, char c));

    /**
     * Get JPEG snapshot picture.
     *
     * @param func A pointer to a callback function.
     *             You can block this function until saving the image datas.
     * @return Status of the error.
     */
    ErrorNumber getJpegSnapshotPicture(void(*func)(char *buf, size_t siz));

    /**
     * Get JPEG preview picture.
     *
     * @param func A pointer to a callback function.
     *             You can block this function until saving the image datas.
     * @return Status of the error.
     */
    ErrorNumber getJpegPreviewPicture(void(*func)(char *buf, size_t siz));

private:
    SerialBuffered serial;
    static const int COMMAND_LENGTH = 6;
    static const int SYNCMAX = 60;
    static const int packageSize = 512;

    ErrorNumber sendInitial(ColorType ct, RawResolution rr, JpegResolution jr);
    ErrorNumber sendGetPicture(PictureType pt);
    ErrorNumber sendSnapshot(SnapshotType st, uint16_t skipFrames);
    ErrorNumber sendSetPackageSize(uint16_t packageSize);
    ErrorNumber sendSetBaudrate(Baud baud);
    ErrorNumber sendReset(ResetType rt, bool specialReset);
    ErrorNumber sendPowerOff();
    ErrorNumber recvData(DataType *dt, uint32_t *length);
    ErrorNumber sendSync();
    ErrorNumber recvSync();
    ErrorNumber sendAck(uint8_t commandId, uint16_t packageId);
    ErrorNumber recvAckOrNck();

    bool sendBytes(char *buf, size_t len, int timeout_us = 20000);
    bool recvBytes(char *buf, size_t len, int timeout_us = 20000);
    bool waitRecv();
    bool waitIdle();
};

#endif
