/**
 ******************************************************************************
 * @file    main.cpp
 * @author  CLab
 * @version V1.0.0
 * @date    2-December-2016
 * @brief   Simple Example application for using the X_NUCLEO_IKS01A1
 *          MEMS Inertial & Environmental Sensor Nucleo expansion board.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/

/* Includes */
#include "mbed.h"
#include "XNucleoIKS01A2.h"
#include <iostream>
#include <cmath>

const float MAX_SPEED = 0.0003f;
const float MIN_SPEED = 0.024095f;
//const float MIN_SPEED = 0.5f;


const float MAX_DEGREES = 10.0f;
const float MIN_DEGREES = 0.0f;



float clamp(float degrees)
{
    if(degrees < MIN_DEGREES) return MIN_DEGREES;
    else if (degrees >= MAX_DEGREES) return MAX_DEGREES;
    else return degrees;
}



float degrees_to_period(float degrees)
{
    return (((MAX_SPEED - MIN_SPEED)/(MAX_DEGREES)) * clamp(std::abs(degrees)) + MIN_SPEED);
}



struct Vec3 {
    float x, y, z;
};

/* Instantiate the expansion board */
static XNucleoIKS01A2 *mems_expansion_board = XNucleoIKS01A2::instance(D14, D15, D4, D5);

/* Retrieve the composing elements of the expansion board */
static LSM303AGRMagSensor *magnetometer = mems_expansion_board->magnetometer;
static LSM6DSLSensor *acc_gyro = mems_expansion_board->acc_gyro;
static LSM303AGRAccSensor *accelerometer = mems_expansion_board->accelerometer;




PwmOut M1_step(D3);
DigitalOut M1_dir(D2);

PwmOut M2_step(D5);
DigitalOut M2_dir(D4);


Serial out(USBTX, USBRX);




Vec3 get_accel()
{
    Vec3 vec3;
    int32_t axes[3];
    acc_gyro->get_x_axes(axes);

    vec3.x = axes[0]/1000.0f + 0.013f;
    vec3.y = axes[1]/1000.0f + 0.004f;
    vec3.z = axes[2]/1000.0f - 0.032f;

    return vec3;
}

void print_vec3(const char* str, const Vec3& vec3)
{
    std::cout << str << vec3.x << " " << vec3.y << " " << vec3.z << "\r\n";
}

float rad_to_deg(float radians)
{
    return radians * 57.2957795131f;
}




int main()
{
    /* Enable all sensors */
    magnetometer->enable();
    accelerometer->enable();
    acc_gyro->enable_x();
    acc_gyro->enable_g();

    Vec3 vec3;

    M1_dir = 1;
    M2_dir = 0;

    M1_step = 0.5f;
    M2_step = 0.5f;

    float angle;
    float period;
    
    float test = 1.05f;

    for(;;) {
        vec3 = get_accel();
        angle = rad_to_deg(atan(vec3.y / sqrtf(vec3.x*vec3.x + vec3.z*vec3.z)));
        period = (degrees_to_period(angle));



//Change direction.
        if(angle > 0) {
            M1_dir = 0;
            M2_dir = 1;
        } else {
            M1_dir = 1;
            M2_dir = 0;
        }
        

        //Control stepper motor.
        out.printf("%f %f %f\r\n", period, angle, degrees_to_period(MAX_DEGREES/test));
        //if(angle < 2.0f || angle > -2.0f)
        M1_step.period(period);
        M2_step.period(period);
        //M1_step.period(0.0003);
        //M2_step.period(0.0003);
        //M1_step.period(degrees_to_period(MAX_DEGREES/test));
        //M2_step.period(degrees_to_period(MAX_DEGREES/test));        
    }
}