/*
   Copyright [2011] [mimilowns@gmail.com]

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
#ifndef SENSECLIENT_H_
#define SENSECLIENT_H_

#include "mbed.h"
#include "HTTPClient.h"
#include "HTTPServer.h"
#include "SenseHandler.h"




// Helper macro to convert two-character hex strings to character value
#define ToHex(Y) (Y>='0'&&Y<='9'?Y-'0':Y-'A'+10)

/**
* SenseClient is an API to interact with Sen.se platform. 
*
* @code
* #include "SenseClient.h"
* #include "picojson.h" //eventualy
*
* SenseClient sense("my sen.se key", "http://proxy:port"); // or "" for no proxy
* string feedID = "a feed id";
* string deviceID = "a device id";
*
* int main() {
*   // init network etc.
* 
*   sense.PostEvent(feedID, tosend);
*   if (sense.Result() == HTTP_OK) {
*        const char* response = sense.ResponseContent().gets();
*        printf("http response content: %s\r\n", response);
*        // ...
*    } else {
*        printf("Sen.se result  / response code : %d / %d\r\n", sense.Result(), sense.Response());
*    }
* @endcode
*/
class SenseClient {

public:
    /** 
    * Creates a client to interacte with sen.se platform.
    *
    * @param senseKey your credential on sen.se platform.
    * @param httpproxy using format http://<ip addr>:<port> or an empty string if using a direct connection.
    */
    SenseClient(const string& senseKey, const string& httpproxy);
    virtual ~SenseClient();

    /** 
    * Posts an event from device to sen.se platform to the given feed. The value must be
    * a valid json string which means that it must be surrounded by double-quotes and other
    * quotes must be escaped etc.
    *
    * Some examples of value:
    * @code
    * string value1 = "\"a text\"";
    * string value2 = "\"10\"";
    * string value3 = "\"{\\"aa\\":\\"tt\\",\\"bb\\":1.660000}\"";
    * @endcode
    * 
    * You can also use picojson library (http://mbed.org/users/mimil/libraries/picojson/lwryzp) as the following:
    * @code
    *   picojson::object v;
    *   v["aa"] = picojson::value(string("tt"));
    *   v["bb"] = picojson::value(double(1.66));
    *   string str = picojson::value(v).serialize();
    *   // if you want to send your data as a json value,
    *   // you have to know that it is in fact a string inside another json
    *   // so it needs to be escaped as any json string
    *   string value4 = picojson::value(str).serialize();
    * @endcode
    *
    * The effective json sent to sen.se platform will be:
    * @code
    * {
    *   "feed_id":4293,
    *   "value":"{\"aa\":\"tt\",\"bb\":1.660000}"
    * }
    * @endcode
    *
    * The response of the call will be available on SenseClient::ResponseContent method and will be
    * a json like this:
    * @code
    * [
    *  {
    *    "publish_id": "5234C4C389AC2",
    *    "value": "{\"aa\":\"tt\",\"bb\":1.660000}",
    *    "timetag": "2011-09-06T12:35:15.559778+00:00",
    *    "feed_id": 4293,
    *    "id": 29768312,
    *    "unit": ""
    *  }
    * ]
    * @endcode
    *
    * @param feedID the feed ID on which you want to post.
    * @param value a valid json string.
    */
    void PostEvent(const string& feedID, const string& value);
    /** 
    * Retrieves the latest event from the given feed.
    *
    * The response of the call will be available on SenseClient::ResponseContent method and will be
    * a json like this:
    * @code
    * {
    *   "publish_id": "5234C4C389AC2",
    *   "value": "{\"aa\":\"tt\",\"bb\":1.660000}",
    *   "timetag": "2011-09-06T12:35:15+00:00",
    *   "feed_id": 4293,
    *   "id": 29768312,
    *   "unit": ""
    * }
    * @endcode
    *
    * @param feedID the feed ID you want to query.
    */
    void GetLastFeedEvent(const string& feedID);
    /** 
    * Retrieves the latest event from the given device.
    *
    * The response of the call will be available on SenseClient::ResponseContent method and will be
    * a json like this:
    * @code
    * {
    *   "publish_id": "5234C4C389AC2",
    *   "value": "{\"aa\":\"tt\",\"bb\":1.660000}",
    *   "timetag": "2011-09-06T12:35:15+00:00",
    *   "feed_id": 4293,
    *   "id": 29768312,
    *   "unit": ""
    * }
    * @endcode
    *
    * @param deviceID the device ID you want to query.
    */
    void GetDeviceLastEvent(const string& deviceID);

    //void GetFeedEvents();
    //void GetDeviceEvents();
    //void GetAccountEvents();


    /**
    * Returns the HTTPClient result of the last HTTP call.
    *
    * @returns The last call result.
    */
    HTTPResult Result();
    /**
    * Returns the HTTP response code of the last HTTP call.
    *
    * @returns The last HTTP code.
    */
    int Response();
    /**
    * Returns the content payload of the last HTTP call.
    *
    * You can use picojson library to parse the content like this:
    * @code
    *    picojson::value v2;
    *    const char* response = sense.ResponseContent().gets();
    *    printf("http response content: %s\r\n", response);
    *    string err = picojson::parse(v2, response, response + strlen(response));
    *    printf("--> res error: %s\r\n", err.c_str());
    *    printf("--> res isarray: %d\r\n", v2.is<picojson::array>());
    *    if(v2.is<picojson::array>()) {
    *        // this is a response from PostEvent
    *        printf("--> res date: %s\r\n", v2.get(0).get("timetag").get<string>());
    *    }
    * @endcode
    *
    * @returns The content payload.
    */
    HTTPText SenseClient::ResponseContent();
    
    /**
    * Returns the given parameter value identified by its name from the given http query string.
    *
    * @return The parameter value 
    * @param queryString The http query string
    * @param name The parameter name
    */
    char* getParam(char *queryString, const char *name);
    
    /**
    * Starts an http server to receive messages from sen.se platform. A function void parseEvent(char* content) MUST be defined in your main.
    *
    * @param port The port on which to start the http server
    */
    void startHttpServer(int port);
    
   

private:
    HTTPText _jsonContent;
    HTTPText _jsonRespContent;
    string _key;
    string _proxy;

    HTTPResult _result;
    int _response;
    
    HTTPServer _svr;


};

#endif // SENSECLIENT_H_