#include "mbed.h"
#include "ctype.h"
#include "TSISensor.h"
#include "MMA8451Q.h"

#define MMA8451_I2C_ADDRESS (0x1d<<1)   // 0x01D<<1 is the default address as specified by the mfg


/********************************
 * GLOBAL VARIABLES / GPIO PINS *
 ********************************/
PwmOut myRedLed(LED1);      //
PwmOut myGreenLed(LED2);    // LED1, LED2 and LED3 are the R, G and B onboard leds
PwmOut myBlueLed(LED3);     //
PwmOut externalLight(PTA1);   // For use with an external led.

MMA8451Q accel(PTE25, PTE24, MMA8451_I2C_ADDRESS);    // 3-axis MEMS accelerometer.
TSISensor touchSlider;      // Capacitive touch sensor, instantiated as touchSlider.
Serial pc(USBTX, USBRX);    // Instantiate a new serial port using the Serial over USB


//-----------------------------------------------------------//
//  Function prototypes for the accelerometer demo function  //
//  as well as the capacitive touch slider demo              //
//-----------------------------------------------------------//
void AccelDemo();
void SliderFadeTimeDemo();
void TouchFadeIn(float val);
void TouchFadeOut(float val);

//-----------------------------------------------------------------------//
//  The following prototypes are all related to the morse code blinking. //
//-----------------------------------------------------------------------//
void morseCodeBlinker(char input[], float speed);
void SendLetter(char input, float speed);
void WordSpace(float speed);
void LetterSpace(float speed);
void Dash(float speed);
void Dot(float speed);


/****************
 * MAIN ROUTINE *
 ****************/
int main()
{
    // Blink the LED's to morse code "HELLO WORLD"
    float blinkSpeed = 0.05f;
    morseCodeBlinker("Hello World", blinkSpeed);

    while (true)
    {
        //--------------------------------------------------------------------------------------//
        //  In the infinite loop of Main, we check to see if the touch slider is being touched  //
        //  and if it is we call the touch slider demo function                                 //
        //  if it is not we will be performing the accelerometer demo function                  //
        //--------------------------------------------------------------------------------------//
        
        char userInput;
    
        if (pc.readable())
        {
            userInput = pc.getc();
            pc.putc(userInput);
            pc.putc('\n');
            pc.putc('\r');
        }
        
        if (touchSlider.readDistance() > 1)
        {
            SliderFadeTimeDemo();
        }
        else
        {
            AccelDemo();
        }
    }

}



/****************
 * SUB ROUTINES *
 ****************/
void SliderFadeTimeDemo()
{
    float val = 0.00f;
    float speedFactor = 0.5f;
    float ledOff = 1.00f;
    int minDistance = 2;

    if (touchSlider.readDistance() > minDistance)
    {
        val = touchSlider.readDistance() * speedFactor;
        TouchFadeIn(val);
        TouchFadeOut(val);
    }
    else
    {
        if (val > minDistance)
        {
            TouchFadeIn(val);
            TouchFadeOut(val);
        }
        else
        {
            myRedLed = ledOff;
            myGreenLed = ledOff;
            myBlueLed = ledOff;
        }
    }
}

void TouchFadeIn(float val)
{   
    for (float i = 1.00f; i > 0.00f; i -= 0.01f)
    {
        myRedLed = i;
        myGreenLed = i;
        myBlueLed = i;
        externalLight = i;
        wait_ms(val);
    }
}


void TouchFadeOut(float val)
{   
    for (float i = 0.00f; i < 1.00f; i += 0.01f)
    {
        myRedLed = i;
        myGreenLed = i;
        myBlueLed = i;
        externalLight = i;
        wait_ms(val);
    }
}

void AccelDemo()
{
    float xAxisData = 1 - abs(accel.getAccX());
    float yAxisData = 1 - abs(accel.getAccY());
    float zAxisData = 1 - abs(accel.getAccZ());

    myGreenLed = xAxisData;
    myRedLed = yAxisData;
    myBlueLed = zAxisData;

    wait_ms(100);
}


void morseCodeBlinker(char input[], float speed)
{
    for (unsigned int i = 0; i < sizeof(input); i++)
    {
        if (input[i] == ' ')
        {
            WordSpace(speed);
        }
        else
        {
            SendLetter(input[i], speed);
            LetterSpace(speed);
        }
    }
}

void SendLetter(char input, float speed)
{
    char lowerCaseLetter = tolower(input);

    switch (lowerCaseLetter)
    {
        case 'a' :
            Dot(speed);
            Dash(speed);
            break;
        case 'b' :
            Dash(speed);
            Dot(speed);
            Dot(speed);
            Dot(speed);
            break;
        case 'c' :
            Dash(speed);
            Dot(speed);
            Dash(speed);
            Dot(speed);
            break;
        case 'd' :
            Dash(speed);
            Dot(speed);
            Dot(speed);
            break;
        case 'e' :
            Dot(speed);
            break;
        case 'f' :
            Dot(speed);
            Dot(speed);
            Dash(speed);
            Dot(speed);
            break;
        case 'g' :
            Dash(speed);
            Dash(speed);
            Dot(speed);
            break;
        case 'h' :
            Dot(speed);
            Dot(speed);
            Dot(speed);
            Dot(speed);
            break;
        case 'i' :
            Dot(speed);
            Dot(speed);
            break;
        case 'j' :
            Dot(speed);
            Dash(speed);
            Dash(speed);
            Dash(speed);
            break;
        case 'k' :
            Dash(speed);
            Dot(speed);
            Dash(speed);
            break;
        case 'l' :
            Dot(speed);
            Dash(speed);
            Dot(speed);
            Dot(speed);
            break;
        case 'm' :
            Dash(speed);
            Dash(speed);
            break;
        case 'n' :
            Dash(speed);
            Dot(speed);
            break;
        case 'o' :
            Dash(speed);
            Dash(speed);
            Dash(speed);
            break;
        case 'p' :
            Dot(speed);
            Dash(speed);
            Dash(speed);
            Dot(speed);
            break;
        case 'q' :
            Dash(speed);
            Dash(speed);
            Dot(speed);
            Dash(speed);
            break;
        case 'r' :
            Dot(speed);
            Dash(speed);
            Dot(speed);
            break;
        case 's' :
            Dot(speed);
            Dot(speed);
            Dot(speed);
            break;
        case 't' :
            Dash(speed);
            break;
        case 'u' :
            Dot(speed);
            Dot(speed);
            Dash(speed);
            break;
        case 'v' :
            Dot(speed);
            Dot(speed);
            Dot(speed);
            Dash(speed);
            break;
        case 'w' :
            Dot(speed);
            Dash(speed);
            Dash(speed);
            break;
        case 'x' :
            Dash(speed);
            Dot(speed);
            Dot(speed);
            Dash(speed);
            break;
        case 'y' :
            Dash(speed);
            Dot(speed);
            Dash(speed);
            Dash(speed);
            break;
        case 'z' :
            Dash(speed);
            Dash(speed);
            Dot(speed);
            Dot(speed);
            break;
    }
}

void WordSpace(float speed)
{
    wait(speed * 7);
}

void LetterSpace(float speed)
{
    wait(speed * 3);
}

void Dash(float speed)
{
    myRedLed = 0;
    myGreenLed = 0;
    myBlueLed = 0;
    wait(speed * 3);
    myRedLed = 1;
    myGreenLed = 1;
    myBlueLed = 1;
    wait(speed);
}

void Dot(float speed)
{
    myRedLed = 0;
    myGreenLed = 0;
    myBlueLed = 0;
    wait(speed);
    myRedLed = 1;
    myGreenLed = 1;
    myBlueLed = 1;
    wait(speed);
}