/*
*   Author:     Alex Millane (millanea@ethz.ch)
*               Matthias Grob
*               Manuel Stalder
*   Date:       04/06/2015
*   Purpose:    Implements nodes which form a localization network.
*               Localization is performed using Decawave DW1000 ranging
*               UWB radios. Additionally initial measurements from the
*               px4 flight controller are used. The project implements
*               three nodes:
*               - Beacon:   Initiates message transmissions, calculates
                            ranges and sends them to the observer node.
                - Anchor:   Received ranging messages and sends calculated
                            time of flight values to the Beacon.
                - Observer: Receives calculated ranges from the beacon and
                            passes them out over a serial connection to a 
                            host performing estimation. Additionally passes
                            mavlink messages out over the same serial connection. 
*/

// Includes
#include "mbed.h"
#include "DW1000.h"
#include "MM2WayRanging.h"
#include "Node.h"
#include "Beacon.h"
#include "Anchor.h"
#include "Observer.h"
#include "debug.h"

// Initializing hardware devices
Serial          serialUSB(USBTX, USBRX) ;           // UART2
Serial          serial1(PA_9,PA_10) ;               // UART1
Serial          serial6(PA_11,PA_12) ;              // UART6

// Observer Defines
Serial& mavlinkIn = serial6 ;
Serial& framesOut = serial1 ;

// Ranging related objects
DW1000          dw(PA_7, PA_6, PA_5, PB_6, PB_9);   // Device driver instanceSPI pins: (MOSI, MISO, SCLK, CS, IRQ)
MM2WayRanging   ranging(dw);                        // Instance of the two way ranging algorithm

// Node factory function
Node& createNode( NodeType nodeType ) ;

// Main programm 
int main() {

    // Setting the baud rates of the serial connections
    serialUSB.baud(57600) ;
    serial1.baud(57600) ;
    serial6.baud(57600) ;

    // Debug start up messages
    debugprintf("Alex Ranging\r\n") ;
    dw.setEUI(0xFAEDCD01FAEDCD01);                                  // basic methods called to check if we have a working SPI connection
    debugprintf("DEVICE_ID register: 0x%X\r\n", dw.getDeviceID());
    debugprintf("EUI register: %016llX\r\n", dw.getEUI());
    debugprintf("Voltage: %fV\r\n", dw.getVoltage());
    
    // Setting the node parameters
    NodeType nodeType   = OBSERVER ; //ANCHOR ; BEACON ; OBSERVER ;
    uint8_t nodeAddress = 5 ;
    
    // Creating the node
    Node& node = createNode( nodeType ) ;
    node.setAddress(nodeAddress) ;
    
    // Executing the application
    while(1){
        // Calling the node functionality
        node.execute() ;
    }
    
}

// Node factory function
Node& createNode( NodeType nodeType ) {
 
    // Creating different node types dependant on argument
    if( nodeType == ANCHOR ){
        Anchor* anchor = new Anchor( ranging, dw ) ;
        return *anchor ;
    }
    else if( nodeType == BEACON ){
        Beacon* beacon = new Beacon( ranging, dw ) ;
        return *beacon ;
    }
    else { // if( nodeType == OBSERVER ){
        Observer* observer = new Observer( ranging, dw, framesOut, mavlinkIn ) ;
        return *observer ;
    }
}