/*  
 * Copyright (c) Eric Tsai 2017
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 * Credit: started with the basic BLE Temperature Beacon code from mbed Bluetooth Low Energy team
 * https://developer.mbed.org/teams/Bluetooth-Low-Energy/code/BLE_TemperatureBeacon/file/0a8bbb6dea16/main.cpp
 *
 * BLE sensor as Beacon advertisements.  Intended to function with specific BLE observer.
 * Tested on nRF51822 targets on mbed.
 * keywords:  todo, tochange
*/


extern "C"
{
   #include "nrf_ecb.h"  //required to call the ecb functions for encryption
}
 
#include "mbed.h"
#include "toolchain.h"
#include "ble/BLE.h"//credit for library @irsanjul:  https://os.mbed.com/users/irsanjul/code/Si7021/

/*******************************************************************************************
 * START tochange: items that may need customization depending on sensors, hardware, and desired behavior
*******************************************************************************************/
const uint16_t Periodic_Update_Seconds = 10; //number of seconds between periodic I/O status re-transmits 900s =15 min.
#define MyDebugEnb 1  //enables serial output for debug, consumes ~1mA when idle
//set I2C pins

//depending on how many sensor values you have, change this and examine Flag_Update_ADV_01
const uint8_t Max_Sensor_Num = 2;  //set to the number of advertisements (temp, humidity, lumins = 3)

//Set beacon rate behavior.  If 4 Seconds_Per_Advertisement @ 300ms ADV_Interval_ms
// then you have about 4 / 0.3 = 13 advertisements for each sensor data.  Probably excessive.
const uint8_t Seconds_Per_Advertisement = 2;    //for each sensor, adverise for this seconds
const uint16_t ADV_Interval_ms = 1000;  //ms between advertisements.  Lower means more beacons. set >=10ms

//sensor data.
char json_char_01[6] = {0, 0, 0, 0, 0, 0}; //hold json for temperature
char json_char_02[6] = {0, 0, 0, 0, 0, 0}; //hold json for humidity
uint8_t json_len_01;
uint8_t json_len_02;

#define MyAnalogPin ADC_CONFIG_PSEL_AnalogInput2

/******************************************************************************************
 * END tochange
*******************************************************************************************/


#if MyDebugEnb
//fail motherfucker
// if you see ~1mA consumption during sleep, that's because MyDebugEnb==1, it's enabled.
Serial device(p1, p2);
#endif
AnalogIn soilSensor(p3); // p9

static Ticker Tic_Stop_Adv;   //used to stop advertising after X seconds
static Ticker Tic_Debounce; //debounce I/O
static Ticker Tic_Periodic; //transmit sensor data on a periodic basis outside I/O events

static bool Flag_Update_IO = false;  //flag to indicate event is hardware interrupt
static bool Flag_Periodic_Call = false;  //flag to indicate event is periodic callback
static bool Flag_Detach_Adv_Tic = false;  //flag to stop advertising
static uint8_t Flag_Update_ADV_01 = 1;  //flag for which sensor data to advertise; 1, 2, 3, etc...

/* Optional: Device Name, add for human read-ability */
const static char     DEVICE_NAME[] = "SL0";


//Advertisement Data
//note:  AdvData[] holds bytes [5] to byte [30] of entire advertising data.  The user content part after ADV flag and header
static uint8_t AdvData[] = {0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0x0};  //26 Bytes manufacturer specific data
char buffer[10]={0, 0, 0, 0, 0, 0, 0, 0, 0, 0}; //hold I/O reading json
char bat_volt_char[6] = {0, 0, 0, 0, 0, 0}; //hold json for battery reading
uint8_t Adv_First_Section[10];  //holds the first several bytes with a pattern indicating this sensor is "one of ours" 
uint8_t mac_reverse[6] = {0x0,0x0,0x0,0x0,0x0,0x0};  //mac address for this module


/*****  Advertisement structure is 31 Bytes  ****************

https://docs.mbed.com/docs/ble-intros/en/latest/Advanced/CustomGAP/

Full Advertisement:
First 5 bytes are set by stack according to flag and header parameters.
Last 26 bytes are user data
-- tabbed --
Byte 0  |   AD1 Length  |       0x02    |   AD1 is 2 bytes long
Byte 1  |   AD1 Type    |       0x01    |   AD1 Data interpreted as flag
Byte 2  |   AD1 Data 0  |       0x06    |   AD1 Data flag mean "00000110"
Byte 3  |   AD2 Length  |       0x1B    |   AD2 is 27 bytes (0x1B) long (rest of this data)
Byte 4  |   AD2 Type    |       0xFF    |   0xFF mean Manufacturer Specific Data
Byte 5  |   AD2 Data 0  |   ADV_Data[0] |   "our device" flag, MAC[3]
Byte 6  |   AD2 Data 1  |   ADV_Data[1] |   "out device" flag, MAC[2]
Byte 7  |   AD2 Data 2  |   ADV_Data[2] |   "out device" flag, MAC[1]
Byte 8  |   AD2 Data 3  |   ADV_Data[3] |   "out device" flag, MAC[0]
Byte 9  |   AD2 Data 4  |   ADV_Data[4] |   battery voltage json MSB, ie 3 in 3.14
Byte 10 |   AD2 Data 5  |   ADV_Data[5] |   battery voltage json
Byte 11 |   AD2 Data 6  |   ADV_Data[6] |   battery voltage json
Byte 12 |   AD2 Data 7  |   ADV_Data[7] |   battery voltage json LSB, ie 4 in 3.14
Byte 13 |   AD2 Data 8  |   ADV_Data[8] |   reserved
Byte 14 |   AD2 Data 9  |   ADV_Data[9] |   reserved
Byte 15 |   AD2 Data 10 |   ADV_Data[10] Encrypted  |   spoof - clock high byte, range 0 to 1800 seconds
Byte 16 |   AD2 Data 11 |   ADV_Data[11] Encrypted  |   spoof - clock low byte
Byte 17 |   AD2 Data 12 |   ADV_Data[12] Encrypted  |   Xmit_Cnt - increments per transmit event, 0-255
Byte 18 |   AD2 Data 13 |   ADV_Data[13] Encrypted  |   JSON[0]
Byte 19 |   AD2 Data 14 |   ADV_Data[14] Encrypted  |   JSON[1]
Byte 20 |   AD2 Data 15 |   ADV_Data[15] Encrypted  |   JSON[2]
Byte 21 |   AD2 Data 16 |   ADV_Data[16] Encrypted  |   JSON[3]
Byte 22 |   AD2 Data 17 |   ADV_Data[17] Encrypted  |   JSON[4]
Byte 23 |   AD2 Data 18 |   ADV_Data[18] Encrypted  |   JSON[5]
Byte 24 |   AD2 Data 19 |   ADV_Data[19] Encrypted  |   JSON[6]
Byte 25 |   AD2 Data 20 |   ADV_Data[20] Encrypted  |   JSON[7]
Byte 26 |   AD2 Data 21 |   ADV_Data[21] Encrypted  |   JSON[8]
Byte 27 |   AD2 Data 22 |   ADV_Data[22] Encrypted  |   JSON[9]
Byte 28 |   AD2 Data 23 |   ADV_Data[23] Encrypted  |   JSON[10]
Byte 29 |   AD2 Data 24 |   ADV_Data[24] Encrypted  |   JSON[11]
Byte 30 |   AD2 Data 25 |   ADV_Data[25] Encrypted  |   JSON[12]

***************************************************/


static uint8_t key[16] = {0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4};
//26 bytes adv data
static uint8_t encrypted[26] = {0x0,0x0,0x0,0x1,0x1,0x1,0x2,0x2,0x2,0x3,0x3,0x3,0x4,0x4,0x4,0x5,0x5,0x5,0x6,0x6,0x6,0x7,0x7,0x7,0x8,0x8};   /* Example of hex data */
//static uint8_t key_buf[16] = {0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4};
static uint8_t key_buf[16] = {0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7, 0x1, 0x2};
static uint8_t src_buf[16] = {0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4};
static uint8_t des_buf[16] = {0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4,0x1,0x2,0x3,0x4};

uint8_t Xmit_Cnt = 1;

/* ****************************************
 * Decides what actions need to be performed on periodic basis
*******************************************/


void stop_adv_Callback(void)
{      
    Flag_Detach_Adv_Tic = true; //flag to stop advertising
    Flag_Periodic_Call = false;
}

void periodic_Callback(void)
{
    #if MyDebugEnb
    device.printf("periodic callback.");
    #endif
    Flag_Update_IO = true;
    Flag_Update_ADV_01 = 1; //flag for advertising first sensor
    Flag_Periodic_Call = true;  //adds the "p" in topic name
    
    Xmit_Cnt++; //increment transmit counter when updating I/O
}

/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}


/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }
    
    /* Set device name characteristic data */
    ble.gap().setDeviceName((const uint8_t *) DEVICE_NAME);

    /* Setup advertising payload */
    //set modes "no EDR", "discoverable" for beacon type advertisements
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    
    //from GAP example
    /* Sacrifice 2B of 31B to AdvType overhead, rest goes to AdvData array you define */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, AdvData, sizeof(AdvData));

    /* Setup advertising parameters:  not connectable */
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_NON_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(ADV_Interval_ms);  //one advertisment every X ms.  Self tickers, so you don't have to worry.
}


/* ****************************************
 * Read battery voltage using bandgap reference
 * shunt Vdd to ADC, thanks to Marcelo Salazar's notes here:
 * https://developer.mbed.org/users/MarceloSalazar/notebook/measuring-battery-voltage-with-nordic-nrf51x/
*******************************************/
uint16_t read_bat_volt(void)
{
    //10 bit resolution, route Vdd as analog input, set ADC ref to VBG band gap
    //disable analog pin select "PSEL" because we're using Vdd as analog input
    //no external voltage reference
    NRF_ADC->CONFIG = (ADC_CONFIG_RES_10bit << ADC_CONFIG_RES_Pos) |
                      (ADC_CONFIG_INPSEL_SupplyOneThirdPrescaling << ADC_CONFIG_INPSEL_Pos) |
                      //(ADC_CONFIG_INPSEL_AnalogInputOneThirdPrescaling << ADC_CONFIG_INPSEL_Pos) |
                      (ADC_CONFIG_REFSEL_VBG << ADC_CONFIG_REFSEL_Pos) |
                      (ADC_CONFIG_PSEL_Disabled << ADC_CONFIG_PSEL_Pos) |
                      //(ADC_CONFIG_PSEL_AnalogInput4 << ADC_CONFIG_PSEL_Pos) |
                      (ADC_CONFIG_EXTREFSEL_None << ADC_CONFIG_EXTREFSEL_Pos);

    //NRF_ADC->CONFIG     &= ~ADC_CONFIG_PSEL_Msk;
    //NRF_ADC->CONFIG     |= ADC_CONFIG_PSEL_Disabled << ADC_CONFIG_PSEL_Pos;
    NRF_ADC->ENABLE = ADC_ENABLE_ENABLE_Enabled;
    NRF_ADC->TASKS_START = 1;
    
    
    //while loop doesn't actually loop until reading comlete, use a wait.
    while (((NRF_ADC->BUSY & ADC_BUSY_BUSY_Msk) >> ADC_BUSY_BUSY_Pos) == ADC_BUSY_BUSY_Busy) {};
    wait_ms(1);

    //save off RESULT before disabling.
    uint16_t myresult = (uint16_t)NRF_ADC->RESULT;
    
    //disable ADC to lower bat consumption
    NRF_ADC->TASKS_STOP = 1;
    NRF_ADC->ENABLE = ADC_ENABLE_ENABLE_Disabled;    //disable to shutdown ADC & lower bat consumption
    
//    return (uint16_t)NRF_ADC->RESULT; // 10 bit
    return myresult;
}  //end read_bat_volt


/* ****************************************
 * Pattern scheme indicating "one of ours"
 * generate first part of ADV data so that observer can recognize it as "one of ours".
 * use specific schema to decide how we're recognizing our sensor ADV
*******************************************/
void hash_first_section(uint8_t * dest, const uint8_t * mac_addr, const char * bat_volt_str)
{
    dest[0] = mac_addr[3];
    dest[1] = mac_addr[2];
    dest[2] = mac_addr[1];
    dest[3] = mac_addr[0];
    dest[4] = bat_volt_str[0];
    dest[5] = bat_volt_str[1];
    dest[6] = bat_volt_str[2];
    dest[7] = bat_volt_str[3];
    dest[8] = 0x10;
    dest[9] = 0x11;
    #if MyDebugEnb
        
        device.printf("hash array: ");
        for (int i=0; i<10; i++)
        {
            device.printf("%x ", dest[i]);
        }
        device.printf("\r\n");
    #endif
}


/* ****************************************
 * 
 * Main Loop
 * 
*******************************************/
int main(void)
{

    #if MyDebugEnb
        device.baud(9600);
        device.printf("started sensor node 36 ");
        device.printf("\r\n");
    #endif

    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);
    
    float bat_reading;  //hold battery voltage reading (Vbg/Vcc)
    
    /* SpinWait for initialization to complete. This is necessary because the
     * BLE object is used in the main loop below. */
    while (ble.hasInitialized() == false) { /* spin loop */ }
    
    //every X seconds, sends period update, up to 1800 (30 minutes)
    Tic_Periodic.attach(periodic_Callback, Periodic_Update_Seconds);  //send updated I/O every x seconds

    //ble.getAddress(0,mac_reverse);  //last byte of MAC (as shown on phone app) is at mac[0], not mac[6];
    ble.gap().getAddress(0,mac_reverse);  //last byte of MAC (as shown on phone app) is at mac[0], not mac[6];
    #if MyDebugEnb
        device.printf("mac = ");
        for (int i=0; i<6; i++) //prints out MAC address in reverse order; opps.
        {
            device.printf("%x:", mac_reverse[i]);
        }
        device.printf("\r\n");
    #endif
    
    
    while (true) 
    {  //Main Loop
        //Update variables holding sensor readings, in preparation for advertisement.
        if (Flag_Update_IO) {
            /* Do blocking calls or whatever hardware-specific action is
             * necessary to poll the sensor. */

            //call attach again on periodic update to reset ticker
            //next periodic updates happens Perioidc_Update_Seconds after I/O events
            Tic_Periodic.attach(periodic_Callback, Periodic_Update_Seconds);   
            //Xmit_Cnt++; //increment transmit counter when updating I/O
            
            //read and convert battery voltage
            bat_reading = (float)read_bat_volt();    
            bat_reading = (bat_reading * 3.6) / 1024.0;
            #if MyDebugEnb
            device.printf("bat reading: %f \r\n", bat_reading);
            #endif
            //write battery voltage
            uint8_t total_chars;
            memset(&bat_volt_char[0], 0, sizeof(bat_volt_char));      //clear out buffer
            //convert battery voltage float value to string reprsentation to 2 decimal places, and save the size of string.
            total_chars = sprintf (bat_volt_char, "%.2f", bat_reading);
                        
            //disable ADC to save power
            NRF_ADC->TASKS_STOP = 1;
            NRF_ADC->ENABLE = ADC_ENABLE_ENABLE_Disabled;    //disable to shutdown ADC & lower bat consumption


            float temptemp = 20;
            
            float temphum = 7;
            
            int humint = -1;
            
            temphum = soilSensor.read();
            humint = soilSensor.read_u16();
            #if MyDebugEnb
            device.printf("tempasd: %f \r\n", temptemp);
            device.printf("hum0: %f \r\n", temphum);
            device.printf("asd: %f \r\n", 3.14f);
            device.printf("hum: %f \r\n", temphum);
            device.printf("hum2: %d \r\n", humint);
            device.printf("this is the end \r\n");
            
            
            
            #endif
            

            memset(&json_char_01[0], 0, sizeof(json_char_01));      //clear out buffer
            memset(&json_char_02[0], 0, sizeof(json_char_02));      //clear out buffer
            json_len_01 = sprintf (json_char_01, "%.2f", temptemp);
            json_len_02 = sprintf (json_char_02, "%.1f", temphum);

            #if MyDebugEnb
            device.printf("char buff: %c%c%c%c \r\n", bat_volt_char[0], bat_volt_char[1], bat_volt_char[2], bat_volt_char[3]);
            device.printf("num chars: %d \r\n", total_chars);
            #endif
            
            Flag_Update_IO = false; //all needed variables for advertisement of sensor data has been updated in memory
        }//end Flag_Update_IO


        //Generate "First Section" for ADV_Data so gateway will recognize our advertisement pattern
        hash_first_section(Adv_First_Section, mac_reverse, bat_volt_char);


        /* ****************************************
            * start writing out ADVData array
            * todo: this is easy to write but hard to read.  Maybe make it easy to read and hard to write?
            ******************************************/
        memset(&AdvData[0], 0, sizeof(AdvData));        //put nulls into adv data
        uint8_t JSON_loc=0; //AdvData[0]

        for(int i = 0; i < 10; i++){
            AdvData[i] = Adv_First_Section[i];          //"our device" flag, MAC[3]
        }

        #if MyDebugEnb
            device.printf("ADV first 10 array: ");
            for (int i=0; i<10; i++)
            {
                device.printf("%x ", AdvData[i]);
            }
            device.printf("\r\n");
        #endif


        JSON_loc = 10;            
        //convert sensor data to string
        for (int i=0; i < json_len_01; i++)
        {
            AdvData[JSON_loc] = json_char_01[i];
            JSON_loc++;
        } //JSON_loc left at location of next character
            
        AdvData[JSON_loc] = 0x2c;    //since AdvData was cleared to start with, we don't need to null term
        JSON_loc++;

        for (int i=0; i < json_len_02; i++)
        {
            AdvData[JSON_loc] = json_char_02[i];
            JSON_loc++;
        } //JSON_loc left at location of next character

        AdvData[JSON_loc] = 0x2c;    //since AdvData was cleared to start with, we don't need to null term
        
        #if MyDebugEnb
            device.printf("AdvData: ");
            for (int i=10; i<JSON_loc+1; i++)  //replace last 16 bytes with encrypted 16 bytes
            {
                device.printf("%x", AdvData[i]);
            }
        #endif

        
        //set payload for advertisement to our custom manufactured data.  First 5 bytes is BLE standard, last 26 bytes is our array
        ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA, AdvData, sizeof(AdvData));
        ble.gap().startAdvertising();
        Tic_Stop_Adv.attach(stop_adv_Callback, Seconds_Per_Advertisement);
        
        if (Flag_Detach_Adv_Tic == true)    //ticker callback flag to stop advertising
        {
        #if MyDebugEnb
            device.printf("stop advertising");
        #endif            
            ble.gap().stopAdvertising();    //may be safer to execute BLE operations in main
            Tic_Stop_Adv.detach();
            Flag_Detach_Adv_Tic = false;
        }
        #if MyDebugEnb
            device.printf("Calling wait for event");
        #endif
        ble.waitForEvent(); //sleeps until interrupt form ticker or I/O
    }
}
