#include "mbed.h"
#include "rtos.h"
#include "USBSerial.h"
#include "max32625pico.h"
#include "SerialInterface.h"

#define USB_MAX_RESP_LENGTH  511
#define USB_MAX_CMD_LENGTH  511

// configure VDDIOH to local 3.3V supply, set dipVio and swdVio to 1.8V supply
MAX32625PICO pico(MAX32625PICO::IOH_3V3, MAX32625PICO::VIO_1V8, MAX32625PICO::VIO_1V8);

// Virtual serial port over USB
USBSerial microUSB(0x0B6A, 0x4360);

DigitalInOut AD[] = {P0_0, P0_1, P0_2, P0_3, P0_4, P0_5, P0_6, P0_7};
DigitalIn RLOS = P1_6;
DigitalIn INTB = P1_7;
DigitalOut WRB = P4_4;
DigitalOut ALE = P4_5;
DigitalOut CSB = P4_6;
DigitalOut RDB = P4_7;

// Serial Interfaces
//I2C i2c(P1_6, P1_7);
//SPI spi(P0_5, P0_6, P0_4);
//DigitalInOut gpio[] = {P0_0, P0_1, P0_2, P0_3, P4_4, P4_5, P4_6, P4_7};
//AnalogIn ain[] = {AIN_0, AIN_1, AIN_2, AIN_3, AIN_4, AIN_5, AIN_6, AIN_7};

// Serial Interface Adapter
//SerialInterface serInt(&i2c, &spi, gpio, ain);

// Threads
Thread threadUSB;

DigitalOut rLED(LED1);
DigitalOut gLED(LED2);
DigitalOut bLED(LED3);

void parseCommand( char* input, char* output);
void write( char* resp );
void read( char* resp );
int _args[MAX_NUM_ARGS];

void usb_thread()
{
    char obuf[USB_MAX_RESP_LENGTH +1];
    char ibuf[USB_MAX_CMD_LENGTH +1];
    int i = 0;

    microUSB.printf("micro USB serial port\r\n");

    while(1) {
        if (microUSB.readable()) {
            ibuf[i]=microUSB.getc();
            if (ibuf[i]!='\r') {
                if (i < USB_MAX_CMD_LENGTH) {
                    i += 1;
                }
            } else {
                bLED = LED_ON;
                if (i < USB_MAX_CMD_LENGTH) {
                    ibuf[i]=0;
//                    microUSB.printf("UART CMD:  %s=", ibuf);
//                    serInt.call(ibuf, obuf);
                    parseCommand( ibuf, obuf );
                    microUSB.printf("%s\r\n", obuf);
                } else {
                    microUSB.printf("[-1]\r\n");
                }
                i=0;
                bLED = LED_OFF;
            }
        }
    }
}


// main() runs in its own thread in the OS
// (note the calls to Thread::wait below for delays)
int main()
{
    rLED = LED_ON;
    gLED = LED_ON;
    bLED = LED_OFF;

//  Configure P4_4 through P4_7 for 3.3V I/O
    pico.vddioh(P0_0, MAX32625PICO::VIO_IOH);
    pico.vddioh(P0_1, MAX32625PICO::VIO_IOH);
    pico.vddioh(P0_2, MAX32625PICO::VIO_IOH);
    pico.vddioh(P0_3, MAX32625PICO::VIO_IOH);
    pico.vddioh(P0_4, MAX32625PICO::VIO_IOH);
    pico.vddioh(P0_5, MAX32625PICO::VIO_IOH);
    pico.vddioh(P0_6, MAX32625PICO::VIO_IOH);
    pico.vddioh(P0_7, MAX32625PICO::VIO_IOH);
    pico.vddioh(P1_6, MAX32625PICO::VIO_IOH);
    pico.vddioh(P1_7, MAX32625PICO::VIO_IOH);
    pico.vddioh(P4_4, MAX32625PICO::VIO_IOH);
    pico.vddioh(P4_5, MAX32625PICO::VIO_IOH);
    pico.vddioh(P4_6, MAX32625PICO::VIO_IOH);
    pico.vddioh(P4_7, MAX32625PICO::VIO_IOH);

    rLED = LED_OFF;

// Start USB serial thread
    threadUSB.start(usb_thread);

// Start main loop
    while(1) {
        Thread::wait(500);
        gLED = !gLED;
    }
}

// taken from SerialInterface.cpp
void parseCommand( char* input, char* output)
{
    char cmd;
    _args[0] = 0; // reset _args
    if (*input == '/') {  // only proceed if input is the format '/w/addr/data'
        input++; // look at second character
        cmd = *input; // cmd should be either 'w' or 'r'
        input = strchr(input, '/'); // strchr returns a pointer to the first occurance of a character, in this case, it is the second '/'
        while (*input == '/') { 
            input++; // advance to next character
            _args[(_args[0]+1)] = strtol(input, &input, 0); // parse data as integer number
            if (input) { // if there is still data, increase
                _args[0]++;
            }
        }
        switch (cmd) {
            case 'w':  // /w/addr/data
                write(output);
                break;
            case 'r':
                read(output);
                break;
            default:
                sprintf(output, "!commands: w r");
                break;
        }
    } else {
        sprintf(output, "!format:  /cmd/arg1/arg2...");

    }
}

void write(char* resp)
{
    int i;
    // _args[1] should be the address
    // _args[2] should be the data to write
    ALE = 1;
    RDB = 1;
    WRB = 1;
    CSB = 0;
    // Set address
    for( i=0; i<8; i++ ) {
        AD[i].output();
        if( ((int)_args[1] & (1<<i)) > 0 ) 
            AD[i] = 1;
        else
            AD[i] = 0;
    }
    wait_us(1.0);
    // ALE falling latches in address
    ALE = 0;
    wait_us(1.0);
    // Set WRB = 0
    WRB = 0;
    wait_us(1.0);
    // Set data
    for( i=0; i<8; i++ ) {
        if( ((int)_args[2] & (1<<i)) > 0 ) 
            AD[i] = 1;
        else
            AD[i] = 0;
    }
    wait_us(1.0);
    // Enable WRB to latch in data
    WRB = 1;
    wait_us(1.0);
    // set CSB
    CSB = 1;
    wait_us(1.0);
    // Set ALE
    ALE = 1;
    wait_us(1.0);
    resp = "[0]";
}
    
void read(char* resp)
{
    int i;
    // _args[1] should be the address
    ALE = 1;
    RDB = 1;
    WRB = 1;
    CSB = 0;
    int readVal=0;
    // Set address
    for( i=0; i<8; i++ ) {
        if( ((int)_args[1] & (1<<i)) > 0 ) 
            AD[i] = 1;
        else
            AD[i] = 0;
    }
    wait_us(1.0);
    // ALE falling latches in address
    ALE = 0;
    wait_us(1.0);
    // Set RDB = 0
    RDB = 0;
    wait_us(1.0);
    // Read data
    // First set as inputs
    for( i=0; i<8; i++ ) AD[i].input();
    wait_us(1.0);
    for( i=0; i<8; i++ ) readVal = readVal + (1<<i) * AD[i];
    sprintf(resp, "%d", readVal);
    wait_us(1.0);
    // Enable WRB to latch in data
    RDB = 1;
    wait_us(1.0);
    // set CSB
    CSB = 1;
    wait_us(1.0);
    // Set ALE
    ALE = 1;
    wait_us(1.0);
}