/** Linear Stepper Motor control library
 *  
 *  @class   LinStepMtr
 *  @author  Mike Marzano
 *  @version 1.0 ()
 *  
 *  This is the driver for A Nema 17 linear stepper motor
 *  
 *  ----------------------IMPORTANT--------------------
 *
 *  ---------------------------------------------------
 */
 
#ifndef MBED_LIN_STEP_MTR
#define MBED_LIN_STEP_MTR
 
#include "mbed.h"

/**********
* Defines *
**********/
#define DEFAULT_RPM 300   // RPM (1rpm = 3.333 steps/sec)
#define MAX_RPM 400       // RPM
#define MIN_RPM 200       // in RPM
#define MAX_DOUBLE_VAL 9223372036854775800
#define MIN_DOUBLE_VAL -9223372036854775800
 
/** Linear Stepper Motor control class
 *
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "lin_step_mtr.h"
 *  
 *  // motor's leads for A,A',B,B' that are hooked upto a dual H-bridge
 *  LinStepMtr mtr(p20, p19, p17, p18);    
 *  
 *  int main() {
 *      // Rotates the motor Clockwise at 400 rpm for 10 revolutions
 *      mtr.set_speed(400);
 *      mtr.rotate(LinStepMtr::CW,10);
 *      
 *      // Rotates the motor Counterclockwise at 200 rpm for 5 revolutions
 *      mtr.set_speed(200);
 *      mtr.rotate(LinStepMtr::CCW, 5);  
 *  }
 *  @endcode
 */

class LinStepMtr {
 
public:
    
    /** Direction Control **/
    typedef enum {
        CW = 0,     // Motor is spinning in CLOCKWISE direction
        CCW = 1,    // Motor is spinning in COUNTER-CLOCKWISE direction
    } Direction;
    
    /** Steps of motor for movement *
      * In form A B A' B'           */
    typedef enum {
            STOP =  0b0000,
            ONE =   0b1100,
            TWO =   0b0110,
            THREE = 0b0011,
            FOUR =  0b1001
        } Step_Num;
    
    /** Create a linear stepper motor object connected to specified DigitalOut pins
     *
     *  @param A_f DigitalOut pin for Forward Control of H-Brigde Port A (AIN1)
     *  @param A_r DigitalOut pin for Reverse Control of H-Brigde Port A (AIN2)
     *  @param B_f DigitalOut pin for Forward Control of H-Brigde Port B (BIN1)
     *  @param B_r DigitalOut pin for Reverse Control of H-Brigde Port B (BIN2)
     */
    LinStepMtr(PinName A_f, PinName A_r, PinName B_f, PinName B_r);
    
    /** Create a linear stepper motor object connected to specified DigitalOut pins
     *
     *  @param A_f DigitalOut pin for Forward Control of H-Brigde Port A (AIN1)
     *  @param A_r DigitalOut pin for Reverse Control of H-Brigde Port A (AIN2)
     *  @param B_f DigitalOut pin for Forward Control of H-Brigde Port B (BIN1)
     *  @param B_r DigitalOut pin for Reverse Control of H-Brigde Port B (BIN2)
     *  @param m_rpm Sets the max speed  in RPM of the motor
     */
    LinStepMtr(PinName A_f, PinName A_r, PinName B_f, PinName B_r, int m_rpm);
    
    /** Destructor
     */
    ~LinStepMtr();
    
    /** Gets the current speed in RPM
     *
     * @return motor's speed in RPM
     */
    float get_speed();
    
    
    /** Sets the value of speed in RPM 
     *
     * @param rpm speed in rpm (maximum value of 400 rpm, min value of 200 rpm)
     */
    void set_speed(float rpm);
    
    /** Gets the number of revolutions since motor was initialized.
     *  Positive means more CW than CCW movement, negative is opposite
     *
     * @return revolution count
     */
    double get_rev();
    
    /** Sets the value of min_rev_cnt. This value is used as a software stop to 
     * prevent the motor from rotating any farther while going counterclockwise.
     * Defaults to the minimum value of a double.
     *
     * @param rc new minimum revolution count limit value
     */
    void set_min_rev_cnt(double rc);
    
    /** Sets the value of max_rev_cnt. This value is used as a software stop to 
     * prevent the motor from rotating any farther while going clockwise
     * Defaults to the maximum value of a double.
     *
     * @param rc new maximum revolution count limit value
     */
    void set_max_rev_cnt(double rc);
    
    /** Gets the value of the current minumum revolution count
     *
     * @return the current minimum revevolution count limit
     */
    double get_min_rev_cnt();
    
    /** Gets the value of the current maximum revolution count
     *
     * @return the current maximum revevolution count limit
     */
    double get_max_rev_cnt();
    
    /** Rests the revolution count limits to max/min of double
    */
    void RESET_rev_cnts();
    
    
    /** Gets the current direction
     *  
     * @return the current direction of motion
     */
    Direction get_dir();
    
    /** NOT SUPORTED - Set the direction
     *
     * @param d new direction of motion
     */
    //void set_dir(Direction d);
    
       
    /** Rotates the motor for a set number of rotations in the given direction
     *
     * @param d direction of rotation
     * @param rev number of revolutions to rotate (defaults to 1 revolution)
     * @return the current revolution count after rotation is complete
     */
    double rotate(Direction d, float rev);
    
    
private:
    /** Private class Step
     *
     * The Linear stepper motor need to have its coils turned on and off in a 
     * precise pattern in order to rotate. Each on/off configuration is called a 
     * step. This class allows for those configurations, which are defined in the
     * StepNum enum, to be easily written to the motor control as well as allows
     * for definition of pre- increment/decrement operators.
     */
    class Step {
     public:     
        //constructior
        Step() : cur_step(ONE) {}; // Default state is step ONE
        
        //increment step pre increment operator
        Step_Num operator++();
        //decrement step pre decrement operator
        Step_Num operator--();
        //Sets step equal to the given step
        void operator=(Step_Num s);
        //getter for cur_step
        Step_Num get_cur_step();
        
     private:
        // the current step.
        Step_Num cur_step;
     
     }; 
    
    //FEATURE TO BE ADDED
    /** Spins up the motor by stepping up from min speed to desired speed
     *
     * @param rpm the ending speed (defaults to current set speed)
     */
    //void spin_up(float rpm=-1);
    
    //FEATURE TO BE ADDED
    /** Spins down the motor by stepping speed down from current speed to min_speed
     *
     * @param rpm the ending speed (defaults to turning motor off)
     */
    //void spin_down(float rpm=-1);
        
    /** Variables **/
    BusOut mtr_ctrl;                    // 4-bit Bus Controlling the H-Brigde
                                        // form A B A' B'
    const int max_speed;                // Software Limit for max motor speed in steps/second
    const float max_rpm;                // Software Limit for max motor speed in rpm
    static const int min_speed  
        = (float)MIN_RPM * 10 / 3;      // Software Limit for min motor speed in steps/sec
    static const int min_rpm = MIN_RPM; // Software Limit for min motor speed in rpm
    volatile int speed;                 // Speed of Rotation (in steps per second)
    volatile double rev_cnt;            // Current Revolution count of motor
    volatile double min_rev_cnt;        // software limit for lowest rev count the moter can reach.
    volatile double max_rev_cnt;        // software limit for highest rev count the moter can reach. 
    Step cur_step;                      // Current Step, i.e. which one was just written to the motor 
    Direction dir;                      // The direction for the motor to run
};
#endif