#include "mbed.h"

struct HVControl {
    int freq, max_freq, min_freq; // oscillator parameters
    float voltage_ref, divider_value, voltage, thresh, safe_voltage; //hv probe parameters
    PinName oscillator_pin, measure_pin;

    int period(int f) {
        double T=0;
        if (f < 1) {
            T = 1;
        } else {
            T = 1000000/f;
        }
        return (int)(T);
    }

    int *hvparams; //array is indexed with the frequency

    HVControl(PinName oscillator_pin_, PinName measure_pin_) {
        max_freq = 10000; //default maximum frequency
        min_freq = 1000; //default minimum frequency
        divider_value = 304; //default board value
        voltage_ref = 3.3; //mbed default voltage reference
        voltage = 0; // actual voltage
        safe_voltage = 20;
        thresh = 1000; //default maximum high voltage threshold
        oscillator_pin = oscillator_pin_;
        measure_pin = measure_pin_;
        freq = max_freq;
    }

    float from_adc_voltage(float adc_v) {
        return adc_v * divider_value * voltage_ref;
    }

    float to_adc_voltage(float voltage_) {
        return voltage_/(divider_value * voltage_ref);
    }

    bool measure(void) {
        AnalogIn probe(measure_pin);
        voltage = from_adc_voltage(probe.read());
        if(voltage > thresh) {
            return 0;
        } else {
            return 1;
        }
    }

    void set_frequency(int f) {
        if(f > max_freq) {
            f = max_freq;
        }
        if(f < min_freq) {
            f = min_freq;
        }
        PwmOut pwm(oscillator_pin);
        int p = period(f);
        freq = f;
        pwm.period_us(p);
        pwm.pulsewidth_us((int)(p/2));
    }

    int freq_to_index(int f) {
        return (f/100)-10;
    }

    int index_to_freq(int f) {
        return (f+10)*100;
    }

    void calibrate(void) {
        hvparams = new int[69];
        shutdown();
        for(int f=7800; f>=1000; f-=100) {
            set_frequency(f);
            wait(1);
            if(measure()) {
                hvparams[freq_to_index(f)]=(int)voltage;
            } else {
                shutdown();
                break;
            }
        }
    }

    void shutdown(void) {
        DigitalOut pwm_(oscillator_pin);
        pwm_.write(1);
        freq = 0;
        measure();
        while(voltage > safe_voltage) {
            measure();
            wait(1);
        }
    }

    float get_voltage(void) {
        measure();
        return voltage;
    }

    int *get_hvparams(void) {
        return hvparams;
    }

    int get_frequency(void) {
        return freq;
    }
};