
/* File inclusion */
#include "mbed.h"
#include "stdint.h"
#include "QEI.h"
#include "project_defines.h"
#include "buttons.h"
#include "encoder_interface.h"
#include "nextion_interface.h"
#include "pressure_sensors.h"
#include "stepper_motor_driver.h"
#include "ventilator.h"
#include "electric_valves.h"

/* Object definition */
Ticker ticker_int_1ms; // Ticker interrupt
DigitalOut  test_pin(LED1);

/* Global variable definition */

volatile uint8_t read_pressure_sensors_flag;
volatile uint8_t pressure_sensor_display_update_flag;
volatile uint8_t pressure_sensor_waveform_update_flag;

uint16_t volume_setpoint = VOLUME_SETPOINT_VALUE_DEFAULT;
uint8_t resp_frequency = RESP_FREQUENCY_VALUE_DEFAULT;
float   inspiration_time = INSPIRATION_TIME_VALUE_DEFAULT;
float   expiration_time = EXPIRATION_TIME_VALUE_DEFAULT;


Main_Screen_State_t main_screen_state = MAIN_SCREEN_DEFAULT_STATE;
uint8_t first_time_in_state = 1;

/* Function definition */

void System_Initialize(void){
    Buttons_Initialize();
    Valves_Initialize();
    Nextion_Interface_Initialize();
    Pressure_Sensors_Initialize();
    ticker_int_1ms.attach(&Ticker_ISR, 0.001);
    Stepper_Driver_Init();
}


void Nextion_Encoder_Interaction(void){

 switch(main_screen_state){
            case Main_Screen_Cursor_Disable:
                if(Get_Button_Press(1 << ENTER_SW)){
                    rotary_encoder.reset();
                    main_screen_state = Main_Screen_Cursor_Enable;
                    first_time_in_state = 1;
                }
                
                break;  
            
            case Main_Screen_Cursor_Enable:
                if(first_time_in_state){
                    first_time_in_state = 0;
                    Parameter_Selection_Box_Update();
                }else if(parameter_selection_index_change_flag){
                    parameter_selection_index_change_flag = 0; 
                    Parameter_Selection_Box_Update();
                }else{
                    //Does nothing
                }
                
                if(Get_Button_Press(1 << ENTER_SW)){
                    main_screen_state = (Main_Screen_State_t)(2 + parameter_selection_index);
                    first_time_in_state = 1;
                }
                
                break;         
            
            
            case Main_Screen_Volume_Setpoint_Adjust:
                if(first_time_in_state){
                    first_time_in_state = 0;
                    nextion_display.printf("t1.pco=%d", ADJUST_COLOR); // Change font color
                    nextion_display.printf("%c%c%c", 0xff, 0xff, 0xff);
                    
                    /* Read pressure sensors if necessary */ 
                    if(read_pressure_sensors_flag){
                        Pressure_Sensors_Read();
                        Calculate_Flow_From_Pressure();
                        Calculate_Volume_From_Flow();
                        read_pressure_sensors_flag = 0;
                    }                     
                    
                    Volume_Setpoint_Display_Update();
                }else if(volume_setpoint_index_change_flag){
                    volume_setpoint_index_change_flag = 0; 
                    Volume_Setpoint_Display_Update();
                }else{
                    //Does nothing
                }
                
                if(Get_Button_Press(1 << ENTER_SW)){
                    main_screen_state = Main_Screen_Cursor_Disable;
                    first_time_in_state = 1;
                    Volume_Setpoint_Fix();
                    Parameter_Selection_Box_Remove();
                }
                
                break;
 
            case Main_Screen_Resp_Frequency_Adjust:
                if(first_time_in_state){
                    first_time_in_state = 0;
                    nextion_display.printf("t4.pco=%d", ADJUST_COLOR); // Change font color
                    nextion_display.printf("%c%c%c", 0xff, 0xff, 0xff);
                    
                    /* Read pressure sensors if necessary */ 
                    if(read_pressure_sensors_flag){
                        Pressure_Sensors_Read();
                        Calculate_Flow_From_Pressure();
                        Calculate_Volume_From_Flow();
                        read_pressure_sensors_flag = 0;
                    }                     
                    
                    Resp_Frequency_Display_Update();
                }else if(resp_frequency_index_change_flag){
                    resp_frequency_index_change_flag = 0; 
                    Resp_Frequency_Display_Update();
                }else{
                    //Does nothing
                }
                
                if(Get_Button_Press(1 << ENTER_SW)){
                    main_screen_state = Main_Screen_Cursor_Disable;
                    first_time_in_state = 1;
                    Resp_Frequency_Fix();
                    Parameter_Selection_Box_Remove(); 
                }
                
                break;         
            
            
            case Main_Screen_I_E_Ratio_Adjust:
                if(first_time_in_state){
                    first_time_in_state = 0;
                    nextion_display.printf("t7.pco=%d", ADJUST_COLOR); // Change font color
                    nextion_display.printf("%c%c%c", 0xff, 0xff, 0xff);
                    
                    /* Read pressure sensors if necessary */ 
                    if(read_pressure_sensors_flag){
                        Pressure_Sensors_Read();
                        Calculate_Flow_From_Pressure();
                        Calculate_Volume_From_Flow();
                        read_pressure_sensors_flag = 0;
                    }                     
                    
                    I_E_Ratio_Display_Update();
                }else if(i_e_ratio_index_change_flag){
                    i_e_ratio_index_change_flag = 0; 
                    I_E_Ratio_Display_Update();
                }else{
                    //Does nothing
                }
                
                if(Get_Button_Press(1 << ENTER_SW)){
                    main_screen_state = Main_Screen_Cursor_Disable;
                    first_time_in_state = 1;
                    I_E_Ratio_Fix();
                    Parameter_Selection_Box_Remove();
                }
                
                break; 
                
            default:
                break;
                
        }   
 }
 
 
 void Nextion_Update_Sensor_Values(void){
    
    static float previous_pressure_02_psi = 0;
    static float previous_volume_ml = 0;
    static float previous_flow_lpm = 0;
    float delta;
    
    /* Check if anu value has change in a meningful way */
    delta = volume_ml - previous_volume_ml;
    if((delta > VOLUME_DISPLAY_THRESHOLD) || (delta < -PRESSURE_DISPLAY_THRESHOLD)){
        Volume_Display_Update();
    } 
    
    delta = pressure_02_psi - previous_pressure_02_psi;
    if((delta > PRESSURE_DISPLAY_THRESHOLD) || (delta < -PRESSURE_DISPLAY_THRESHOLD)){
        Pressure_Display_Update();
    }
    
    delta = flow_lpm - previous_flow_lpm;
    if((delta > FLOW_DISPLAY_THRESHOLD) || (delta < -PRESSURE_DISPLAY_THRESHOLD)){
        Flow_Display_Update();
    }    
    
    
    /* Update the previous values */
    previous_volume_ml = volume_ml;
    previous_pressure_02_psi = pressure_02_psi;
    previous_flow_lpm = flow_lpm;

 }
 
 
void Nextion_Update_Waveform_Values(void){
    Nextion_Plot_Volume_Waveform();  
    Nextion_Plot_Pressure_Waveform();  
    Nextion_Plot_Flow_Waveform(); 
}


// Ticker Interrupt Service Routine
void Ticker_ISR(void){
   
    static uint8_t debounce_conta_1ms = 0;
    static uint8_t encoder_conta_1ms = 0;
    static uint8_t sensor_conta_1ms = 0;
    static uint8_t sensor_waveform_conta_1ms = 0;
    static uint16_t sensor_display_conta_1ms = 0;
     
    debounce_conta_1ms++;
    encoder_conta_1ms++;
    sensor_conta_1ms++;
    sensor_waveform_conta_1ms++;
    sensor_display_conta_1ms++;
    
    
    
    // Read buttons and performs debounce action
    if(debounce_conta_1ms == DEBOUNCE_PERIOD_MS){
        debounce_conta_1ms = 0;
        Button_Debounce();
    }    
    // Read encoder and update the proper indexes    
    if(encoder_conta_1ms == ENCODER_READ_PERIOD_MS){
        encoder_conta_1ms = 0;
        Encoder_Read();
    }        
    // Indicates if it is time to acquire a new sample from the sensors
    if(sensor_conta_1ms == PRESSURE_SENSOR_READ_PERIOD_MS){
        sensor_conta_1ms = 0;
        read_pressure_sensors_flag = 1;
    }       
    // Indicates if it is time to plot the volume, flow and pressure values */   
    if(sensor_waveform_conta_1ms  == SENSOR_WAVEFORM_PLOT_PERIOD_MS){
        sensor_waveform_conta_1ms  = 0;
        pressure_sensor_waveform_update_flag = 1;
        test_pin = !test_pin;
    }    
    // Indicates if it is time to display the volume, flow and pressure values */   
    if(sensor_display_conta_1ms  == SENSOR_DISPLAY_PERIOD_MS){
        sensor_display_conta_1ms  = 0;
        //pressure_sensor_display_update_flag = 1;
    }               
        
}
 
        