
/* File inclusion */
#include "mbed.h"
#include "stdint.h"
#include "math.h"
#include "project_defines.h"
#include "pressure_sensors.h"
#include "filters.h"

/* Object definition */
I2C pressure_sensor_01(PRESSURE_SENSOR_01_SDA_PIN, PRESSURE_SENSOR_01_SCL_PIN);
I2C pressure_sensor_02(PRESSURE_SENSOR_02_SDA_PIN, PRESSURE_SENSOR_02_SCL_PIN);

/* Global variable definition */
float pressure_01_psi, pressure_02_psi;
float flow_lpm, volume_ml;

char sensor_read_data_buffer[2];

volatile uint8_t peak_volume_flag = 0;

/* Function definition */

void Pressure_Sensors_Initialize(void){
    
    pressure_sensor_01.frequency(PRESSURE_SENSOR_I2C_CLK_HZ); 
    pressure_sensor_02.frequency(PRESSURE_SENSOR_I2C_CLK_HZ);
    
    /* Initialize filters */
     Pressure_01_Cheby2_LPF(0, 1);
     Pressure_02_Cheby2_LPF(0, 1);
     Flow_Cheby2_HPF(0, 1);
     Flow_FIR_LPF(0,1);
}


void Pressure_Sensors_Read(void){
    
    int16_t pressure_01_bin;
    int16_t pressure_02_bin;
    
    /* Read pressure 1 (differential) binary value */
    
    sensor_read_data_buffer[0] = 0;
    sensor_read_data_buffer[1] = 0;
    
    pressure_sensor_01.read(PRESSURE_SENSOR_01_SRA, sensor_read_data_buffer, 2);
    pressure_01_bin = ((sensor_read_data_buffer[0] << 8) | sensor_read_data_buffer[1]) & 0x3FFF;
    if(pressure_01_bin < PRESSURE_SENSOR_OUTPUT_MIN){
        pressure_01_bin = PRESSURE_SENSOR_OUTPUT_MIN;
    }     
    
    /* Read pressure 2 binary value */
    
    sensor_read_data_buffer[0] = 0;
    sensor_read_data_buffer[1] = 0;
    
    pressure_sensor_02.read(PRESSURE_SENSOR_02_SRA, sensor_read_data_buffer, 2);
    pressure_02_bin = ((sensor_read_data_buffer[0] << 8) | sensor_read_data_buffer[1]) & 0x3FFF;
    if(pressure_02_bin < PRESSURE_SENSOR_OUTPUT_MIN){
        pressure_02_bin = PRESSURE_SENSOR_OUTPUT_MIN;
    }    
    
    /* Calculate pressure values in PSI */
    pressure_01_psi = (pressure_01_bin - PRESSURE_SENSOR_OUTPUT_MIN) * (PRESSURE_SENSOR_PRESSURE_MAX - PRESSURE_SENSOR_PRESSURE_MIN);
    pressure_01_psi = (pressure_01_psi / (PRESSURE_SENSOR_OUTPUT_MAX - PRESSURE_SENSOR_OUTPUT_MIN)) + PRESSURE_SENSOR_PRESSURE_MIN;
    pressure_01_psi += PRESSURE_SENSOR_01_OFFSET_PSI;
    
    pressure_02_psi = (pressure_02_bin - PRESSURE_SENSOR_OUTPUT_MIN) * (PRESSURE_SENSOR_PRESSURE_MAX - PRESSURE_SENSOR_PRESSURE_MIN);
    pressure_02_psi = (pressure_02_psi / (PRESSURE_SENSOR_OUTPUT_MAX - PRESSURE_SENSOR_OUTPUT_MIN)) + PRESSURE_SENSOR_PRESSURE_MIN;
    pressure_02_psi += PRESSURE_SENSOR_02_OFFSET_PSI;
    
    /* Filter pressure values */
    pressure_01_psi = Pressure_01_Cheby2_LPF(pressure_01_psi, 0);
    pressure_02_psi = Pressure_02_Cheby2_LPF(pressure_02_psi, 0);
    
} 


void Calculate_Flow_From_Pressure(void){
    
    float diff_pressure_psi;
    static float prev_flow_lpm = 0;
    
    diff_pressure_psi = pressure_02_psi - pressure_01_psi;    
    
    if(diff_pressure_psi < 0){
        flow_lpm = -SPIROMETER_SENSOR_CONSTANT_OXYGEN * sqrt(-diff_pressure_psi);    
    }else{
        flow_lpm = SPIROMETER_SENSOR_CONSTANT_OXYGEN * sqrt(diff_pressure_psi);  
    }    
    
    /* Filter flow value */
    flow_lpm = Flow_FIR_LPF(flow_lpm, 0);
    flow_lpm = Flow_Cheby2_HPF(flow_lpm, 0);
    
    if((prev_flow_lpm > 0) && (flow_lpm <= 0)){
        peak_volume_flag = 1;
    }
    
    prev_flow_lpm = flow_lpm;
    
}


/* Simpson's integration */

void Calculate_Volume_From_Flow(void){

    static uint8_t integral_count = 0;
    static float f_x[3] = {0.0, 0.0, 0.0};
    float delta_volume_ml;
    
    f_x[2] = f_x[1];
    f_x[1] = f_x[0];
    f_x[0] = flow_lpm;
    
    integral_count++;
    
    if(integral_count >= 2){
        delta_volume_ml = ((float)PRESSURE_SENSOR_READ_PERIOD_MS / 180.0) * (f_x[2] + (4 * f_x[1]) + f_x[0]);
        volume_ml += delta_volume_ml;
        integral_count = 0;
    }
    
}


